/*=======================================================================================

 M95XXX: 32M Bits Serial SPI Bus EEPROM Verilog Simulation Model

 =========================================================================================

 This program is provided "as is" without warranty of any kind, either
 expressed or implied, including but not limited to, the implied warranty
 of merchantability and fitness for a particular purpose. The entire risk
 as to the quality and performance of the program is with you. Should the
 program prove defective, you assume the cost of all necessary servicing,
 repair or correction.

 Copyright 2004, 2018, STMicroelectronics Corporation, All Right Reserved.

 =======================================================================================*/

`include "M95XXX_Parameters.v"
//This defines the parameter file for M95P32 process
//Any other M95xxx memory should define here the proper M95xxx parameter file

//=====================================
// M95P32 memory simulation model
//=====================================
module M95XXX(
		C,
		D,
		Q,
		S,
		W,
		HOLD,
		VCC,
		VSS
	);

//=====================================
// I/O signal definition
//=====================================
	input  C,S,VCC,VSS;
	inout D,W,HOLD;
	output Q;

//pragma protect
//pragma protect begin

//===================================================================
//define "flag variable", reflecting the device operation status
//===================================================================
	integer i,page_no,idx;

	reg Q,dout,power_on,power_off,power_on_rst;
	reg clk_in,select_ok,bit_counter_en,bit_counter_ld;
	reg wr_protect,byte_ok,write_data_byte_ok;
	reg write_en_id,write_dis_id,read_data_bytes,fast_read_data_bytes ,fd_read_data_bytes,fq_read_data_bytes,read_id_data_bytes,write_data_in,page_program_data_in,page_program_data_end ,write_id_data_in,write_id_data_end,write_data_end;
	reg sr_write,new_sr_byte,write_new_sr_ok,set_new_lockid_ok;
	reg instruction_byte,address_h_byte,address_h2_byte,address_l_byte,data_byte,lock_byte;
	reg hold_condition,hold_start_next,hold_stop_next;
	reg pageidlocked,readidwrap;
	reg nonxt_byte;
	reg wrapping;
	reg rds_inprogress,jedid_inprogress,read_config_safe_inprogress;
	reg writemem_in_progress = 0;

	reg[1:0] mode;
	reg[4:0] operation;
	reg[7:0] instruction;
	reg[7:0] status_reg,shift_in_reg,data_out_buf,bit_counter,safe_reg, config_reg;
	reg[7:0] lock_latch,sr_latch,instruction_code,address_h2_code,address_h_code,address_l_code;
	reg[`MEM_ADDR_BITS-1:0] memory_address;
	reg[`PAGE_ADDR_BITS-1:0] page_address;
	reg[`PAGE_OFFSET_BITS-1:0] page_offset_addr;
	reg[`PAGE_OFFSET_BITS-1:0] start_address,end_address;
	reg[`DATA_BITS-1:0] memory[`MEM_SIZE-1:0];
	reg[`DATA_BITS-1:0] data_latch[`PAGE_SIZE-1:0];
	reg[`DATA_BITS-1:0] memory_id[(`PAGE_SIZE*2)-1:0];
	reg[`PAGE_OFFSET_BITS:0] page_size;

//PAGE PROGRAM BUFFER MODE
	reg page_program_buff_mode;
	reg[`DATA_BITS-1:0] mem_buffer[`PAGE_SIZE-1:0]; //buffer da 512 bytes
	reg[`MEM_ADDR_BITS-1:0] mem_addr_buffer; //buffer per l'indirizzo di memoria
	reg addr_buffer_full;
	reg mem_buffer_full;
	reg page_program_buff_mode_end ;
	reg page_prog_buff_mode_start;
	reg page_program_buff_data_in;
	reg page_program_buff_data_end;

	reg no_wait = 1'b0;

	reg[`PAGE_ADDR_BITS-1:0] page_address_buff;
	reg data_byte_buff;
	reg[`PAGE_OFFSET_BITS-1:0] start_address_buff,start_address_copy,end_address_buff;
	reg[`PAGE_OFFSET_BITS-1:0] page_offset_addr_buff;
	reg[`MEM_ADDR_BITS-1:0] memory_address_buff;

	reg start_PGPR;
	integer num_bytes,incr,page_no_buff;

//===============JEDEC
	reg new_jedec_ok, set_new_jedec_ok;
	reg[7:0] jedec_latch;
	reg[7:0] STManufacturer = 8'b00100000;
	reg[7:0] SPI_Family_code = 8'b00000000;
	reg[7:0] Mem_density_code = 8'b00010110;
	reg [23:0] out_jedec; //{STManufacturer,SPI_Family_code,Mem_density_code}
	reg [15:0] out_conf_safe_reg; // {configuration_register, Safety_register}
	reg clear_safe_reg_en;

//================ SECTOR ERASE internal signals
	reg set_new_sect_eras_ok;
	reg[`MEM_ADDR_BITS-1:0] start_address_sect;
	reg[`PAGE_ADDR_BITS-1:0] page_address_sect;
	reg sect_erase_en, data_byte_sect;
	reg sector_inprogress = 0;
	integer address_no,page_start_sect,page_end_sect;
	integer no_address_start_sect,no_address_end_sect,page_current,sector_no;
	integer k = 0;


//===============================================
//----------WRITE VOLATILE REGISTER internal signal
//===============================================
	reg vol_reg_en,new_volatile_byte;
	reg[7:0] volatile_latch;
	reg[7:0] volatile_reg;

//===============================================
//----------READ VOLATILE REGISTER internal signal
//===============================================
	reg rdv_inprogress = 0;

//===============================================
//--------- FAST READ IDENTIFICATION PAGE internal signal -----------//
//===============================================
	reg fast_read_id_data_bytes;

//===============================================
//----------BLOCK ERASE internal signal
//===============================================
	reg set_new_block_eras_ok,data_byte_block,block_erase_en;
	reg block_inprogress = 0;
	reg[`MEM_ADDR_BITS-1:0] start_address_block;
	reg[`PAGE_ADDR_BITS-1:0] page_address_block;
	integer no_address_start_block,no_address_end_block,page_end_block,page_start_block,page_current_block,address_no_block,block_no,sector_start,sector_end;

//===============================================
//----------CHIP ERASE internal signal
//===============================================
	reg chip_erase_en;
	reg[`MEM_ADDR_BITS-1:0] start_address_chip;

//===============================================
//----------PAGE ERASE internal signal
//===============================================
	reg set_new_page_eras_ok, data_byte_page, page_erase_en;
	reg page_inprogress = 0;
	reg[`PAGE_ADDR_BITS-1:0] page_address_page;
	reg[`MEM_ADDR_BITS-1:0] start_address_page;
	integer no_address_start_page,no_address_end_page;
//===============================================
//----------READ SFDP internal signal
//===============================================
	reg[`DATA_BITS-1:0] sfdp_register[(`PAGE_SIZE)-1:0];
	reg sfdp_data_bytes;

//==============================================================
//----------ENABLE_RESET & SOFTWARE_RESET internal signal
//==============================================================
	reg reset_en_id,reset_en_bit,res_soft_en;
	reg modify_operation = 1'b0;
	reg modify_operation_cher = 1'b0;
	time t_chip_erase,t_modify,t_wait_reset;

//==============================================================
//---------- WRITE_STATUS_REGISTER MODIFIED
//==============================================================
	reg[15:0] shift_in_reg_new;
	reg[7:0] bit_counter_wr;
	reg bit_counter_ld_wr, bit_counter_en_wr,byte_ok_wr;
	reg[7:0] shift_config,config_latch_new;

//==============================================================
//----------DEEP_POWER-DOWN & RELEASE DEEP_POWER-DOWN internal signal
//==============================================================
	reg dpd_en,rdpd_en;
	reg deep_pwr_active;

//===============================================
//define variable regarding timing check
//===============================================
	reg  din_change,r_S,f_S,r_C,r_Cr,f_C,r_H,f_H;
	real t_rS,t_fS,t_rC,t_fC,t_d,t_rC1,t_rH,t_fH,Tc;
	real tCH,tCL,tSLCH,tCHSL,tDVCH,tCHDX,tCHSH,tSHCH,tSHSL,tCLHH,tCLHL,tHLCH,tHHCH;

//==============================================
// Supply & Frequency Check 
//==============================================

real freq_clk;
real time_clk;
real t0_clk = 0.0;
real t1_clk = 0.0;
real tDiff_clk = 0.0;
integer start_time_clk_check = 0; 

//=========================================================
//define parameter regarding instruction
//=========================================================
	parameter WRITE_ENABLE           = 8'b0000_0001;
	parameter WRITE_DISABLE          = 8'b0000_0010;
	parameter READ_STATUS_REGISTER   = 8'b0000_0011;
	parameter WRITE_STATUS_REGISTER  = 8'b0000_0100;
	parameter READ_FROM_MEMORY_ARRAY = 8'b0000_0101;
	parameter WRITE_TO_MEMORY_ARRAY  = 8'b0000_0110;
	parameter READ_ID_PAGE           = 8'b1000_0011;
	parameter WRITE_ID_PAGE          = 8'b1000_0010;
//aggiungo le nuove istruzioni
	parameter FAST_READ_DATA         = 8'b0000_0111; //fast read single output with 1 dummy byte
	parameter FAST_DUAL_READ_DATA    = 8'b0000_1000; //fast read dual output with 1 dummy byte
	parameter FAST_QUAD_READ_DATA    = 8'b0000_1001; //fast read quad output with 1 dummy byte
	parameter PAGE_PROGRAM           = 8'b0000_1010; //page program
	parameter PAGE_ERASE             = 8'b0000_1011; //page erase
	parameter SECTOR_ERASE           = 8'b0000_1100; //sector erase
	parameter BLOCK_ERASE            = 8'b0000_1101; //block erase
	parameter CHIP_ERASE             = 8'b0000_1110; //chip erase
	parameter FAST_READ_ID           = 8'b0000_1111; //fast read identification id
	parameter DEEP_POWER_ENTER       = 8'b0001_0000; //deep power - down enter
	parameter RELEASE_DEEP_POWER     = 8'b0001_0001; //deep power down release (wake up)
	parameter JEDID                  = 8'b0001_0010; //JEDEC identification SF
	parameter READ_CONFIG_SAFETY_REG = 8'b0001_0011; //read configuration and safety register
	parameter READ_VOLATILE_REG      = 8'b0001_0100; //read volatile register
	parameter WRITE_VOLATILE_REG     = 8'b0001_0101; //write volatile register
	parameter CLEAR_SAFETY_STI_FLAGS = 8'b0001_0110; //clear safety sticky flags
	parameter READ_SFDP              = 8'b0001_0111; //read SFDP
	parameter RESET_ENABLE           = 8'b0001_1000; //enable reset
	parameter RESET_SOFTWARE          = 8'b0001_1001; //software reset
//=========================================================
//define parameter regarding operation
//=========================================================
	parameter EN_WRITE               = 5'b00001; //write enable
	parameter DIS_WRITE              = 5'b00010; //write disable
	parameter READ_SR_OUT            = 5'b00011; //read status register
	parameter WRITE_SR_IN            = 5'b00100; //write status register
	parameter READ_DATA_OUT          = 5'b00101; //read data single output
	parameter WRITE_DATA_IN          = 5'b00110; //page write
	parameter READ_ID_DATA_OUT       = 5'b00111; //read identification id
	parameter WRITE_ID_DATA_IN       = 5'b01000; //write identification id
	parameter LOCKID                 = 5'b01001; //LOCKS THE ID PAGE IN READ ONLY MODE
//aggiungo un bit e le nuove operations
	parameter FREAD            = 5'b01010; //fast read single output with 1 dummy byte
	parameter FDREAD           = 5'b01011; //fast read dual output with 1 dummy byte
	parameter FQREAD           = 5'b01100; //fast read quad output with 1 dummy byte
	parameter PGPR             = 5'b01101; //page program
	parameter PGER             = 5'b01110; //page erase
	parameter ER_SECT          = 5'b01111; //sector erase
	parameter BKER             = 5'b10000; //block erase
	parameter CHER             = 5'b10001; //chip erase
	parameter FRDID            = 5'b10010; //fast read identification id
	parameter DPD              = 5'b10011; //deep power - down enter
	parameter RDPD             = 5'b10100; //deep power down release (wake up)
	parameter JEDEC_ID         = 5'b10101; //JEDEC identification SF
	parameter RDCR             = 5'b10110; //read configuration and safety register
	parameter RDVR             = 5'b10111; //read volatile register
	parameter WRVR             = 5'b11000; //write volatile register
	parameter SAFE_CLEAR_FLAG  = 5'b11001; //clear safety sticky flags
	parameter RDSFDP           = 5'b11010; //read SFDP
	parameter RSTEN            = 5'b11011; //enable reset
	parameter RESET            = 5'b11100; //software reset
	parameter PGPR_BUFF        = 5'b11101; //page_program_buffer_mode

//=========================================================
//define parameter regarding device mode
//=========================================================
	parameter device_no_power_mode   = 2'b00;
	parameter active_power_mode      = 2'b01;
	parameter device_standby_mode    = 2'b10;
//=================================
// Aggiunto per Deep power mode!
	parameter deep_power_mode        = 2'b11;
//================================

//==========================
//Fast quad read internal signals
//==================================
	reg dummy_byte = 0;
	reg Qout_en = 0; //quad output port enable signal. if 0  it has input ports, if 1 it has output ports
	reg Qout_en_dual = 0; //dual output port enable signal.

	reg DQ0=1'bz; //D
	reg DQ2=1'bz; //W
	reg DQ3=1'b1; //HOLD output register.

////=============================PAGE PROGRAM INTERNAL SIGNALS===============
	reg[7:0]erased_status_chk;
	reg[`PAGE_OFFSET_BITS-1:0] start_address_chk,end_address_chk;
////=====================================================

//====================================================
//INOUT PORTS assignments
//====================================================
	assign D = (!(Qout_en || Qout_en_dual)) ? 1'bZ : DQ0; //if FQREAD or FDREAD is enabled. the D is DQ0
	// Q is DQ1; and it is always an output!
	assign W = (!Qout_en) ? 1'bZ : DQ2;  //if FQREAD is enabled
	assign HOLD = (!Qout_en) ? 1'bZ : DQ3; //if FQREAD is enabled

//===============================================
//"variable" initialization
//===============================================
	initial
	begin
		mode          = device_no_power_mode;
		power_on      = 1'b0;
		power_off     = 1'b1;
		power_on_rst  = 1'b0;

	end

//==============================================
// Supply & Frequency Check 
//==============================================

	always @(posedge C)
	begin
		if (start_time_clk_check > 1) begin
			if ( `Vcc >= 1.6 && `Vcc < 2.6 ) //low voltage => Max freq = 40MHz
			begin
				t1_clk = $realtime(); 
				tDiff_clk = t1_clk - t0_clk;
				if (tDiff_clk < 25.0) begin  //frequency > 40 MHz 
					$display("\n\n \t\t\t\t\t*************************************\n \t\t\t\t\t*************************************\n \t\t\t\t\t*************************************");
					$display(" *************************************************************************************************************************\n *******************                                     ERROR                                **************************** \n");
					$display(" ************ With this Vcc, Max Working Frequence (Fc) is 40MHz, Clock Frequency inserted is %g MHz ****************",1000.0/tDiff_clk);
					$display(" \t\t\t\t\t*************************************\n \t\t\t\t\t*************************************\n \t\t\t\t\t*************************************\n");

					$finish;
				end 
				else begin
				t0_clk = t1_clk;
				end

			end
			else if ( `Vcc >= 2.6 && `Vcc <= 3.6 ) //high voltage => Max Freq = 80MHz
			begin
				t1_clk = $realtime(); 
				tDiff_clk = t1_clk - t0_clk;
				if (tDiff_clk < 12.5) begin  //frequency > 80 MHz 
					$display("\n\n \t\t\t\t\t*************************************\n \t\t\t\t\t*************************************\n \t\t\t\t\t*************************************");
					$display(" *************************************************************************************************************************\n *******************                                     ERROR                                **************************** \n");
					$display(" ************ With this Vcc, Max Working Frequence (Fc) is 80MHz, Clock Frequency inserted is %g MHz ****************",1000.0/tDiff_clk);
					$display(" \t\t\t\t\t*************************************\n \t\t\t\t\t*************************************\n \t\t\t\t\t*************************************\n");

					$finish;
				end 
				else begin
				t0_clk = t1_clk;
				end			
			end
		end
		else begin
		start_time_clk_check = start_time_clk_check + 1;
		t0_clk = $realtime;
		end
	end
//================================================================================
	


//--------------------------------
	always@(power_on_rst)
	begin
		if(power_on_rst == 1'b1)
		begin
			mode                = device_standby_mode;
			power_on_rst        = 1'b0;
			byte_ok             = 1'b0;
			select_ok           = 1'b0;
			status_reg          = 8'b00; //|  SRWD |  TB  |  x   | BP2 |  BP1  |  BP0  |  WEL  |   WIP  |
			safe_reg            = 8'b00; //| PAMAF |  PUF | ERF  | PRF | ECC1C | ECC2C | ECC3D | ECC3DS |
			config_reg          = 8'h20; //|   x   | DRV1 | DRV2 |  x  |   x   |  x    |   x   |  LID   | // h20(as delivery state)
			operation           = 5'b00000;
			instruction         = 5'b00000;
			write_data_byte_ok  = 1'b0;
			bit_counter_en      = 1'b0;
			write_en_id         = 1'b0;
			write_dis_id        = 1'b0;
			instruction_byte    = 1'b0;
			address_h_byte      = 1'b0;
			address_h2_byte     = 1'b0;
			address_l_byte      = 1'b0;
			data_byte           = 1'b0;
			read_data_bytes     = 1'b0;
			fast_read_data_bytes = 1'b0;
			fd_read_data_bytes  = 1'b0;
			fq_read_data_bytes  = 1'b0;
			read_id_data_bytes  = 1'b0;
			fast_read_id_data_bytes = 1'b0;
			write_data_in       = 1'b0;
			write_data_end      = 1'b0;
			page_program_data_in  = 1'b0;
			page_program_data_end = 1'b0;
///
			page_prog_buff_mode_start = 1'b0;
			page_program_buff_mode = 1'b0;
			page_program_buff_mode_end = 1'b0;
			page_program_buff_data_in = 1'b0;
			page_program_buff_data_end = 1'b0;
			no_wait = 1'b0;

			addr_buffer_full = 1'b0;
			mem_buffer_full = 1'b0;
			start_PGPR = 1'b0;
///
			write_id_data_in    = 1'b0;
			write_id_data_end   = 1'b0;
			wr_protect          = 1'b0;
			data_byte_sect      = 1'b0;
			set_new_sect_eras_ok= 1'b0;
			hold_condition      = 1'b0;
			hold_stop_next      = 1'b0;
			hold_start_next     = 1'b0;
			lock_byte           = 1'b0;
			set_new_lockid_ok   = 1'b0;
			pageidlocked        = 1'b0;
			page_size           = `PAGE_SIZE;
			nonxt_byte          = 1'b0;
			wrapping            = 1'b0;
			rds_inprogress      = 1'b0;
			set_new_jedec_ok    = 1'b0;
			new_volatile_byte   = 1'b0; //////
			jedid_inprogress    = 1'b0;
			read_config_safe_inprogress = 1'b0;
			clear_safe_reg_en   = 1'b0;
			vol_reg_en          = 1'b0;
			volatile_reg        = 8'b00000001; //|  x  |  x  |  x   |   x  |   x   |   x   |  BUFEN  |  BUFLD  |
			///////---------- BLOCK ERASE INSTRUCTION
			set_new_block_eras_ok = 1'b0;
			data_byte_block     = 1'b0;
			//////------------ CHIP ERASE INSTRUCTION
			chip_erase_en       = 1'b0;
			//////------------ READ SFDP_REGISTER INSTRUCTION
			sfdp_data_bytes     = 1'b0;
			//////---- PAGE ERASE INSTRUCTION
			set_new_page_eras_ok = 1'b0;
			data_byte_page     = 1'b0;
			//////----------- WRITE STATUS AND CONFIGURATION REGISTERS INSTRUCTION
			shift_in_reg_new = 16'b0000000000000000;
			//////------------ RESET ENABLE & SOFTWARE RESET INSTRUCTIONS
			reset_en_id          = 1'b0;
			reset_en_bit     = 1'b0;
			res_soft_en          = 1'b0;
			modify_operation     =  1'b0;
			modify_operation_cher =1'b0;
			///////------------ DEEP POWER-DOWN &  DEEP POWER-DOWN RELEASE INSTRUCTIONS
			dpd_en            = 1'b0;
			deep_pwr_active   = 1'b0;
			rdpd_en           = 1'b0;
			//=========================For Frequency check
			start_time_clk_check = 0; 
		end
		$display("%t MEMORY: Reset instruction ended",$realtime);
	end

//==========================================
//Write Protect (Hardware Protected)
//==========================================
	always@(W)
	begin
		if((W == 1'b0)&&(S == 1'b1)&&(power_on == 1'b1) && safe_reg[6] == 0)  //for PUF = 0;
			wr_protect = 1'b1;

		else if((W == 1'b0)&&(S == 1'b1)&&(power_on == 1'b1) && safe_reg[6] == 1) //for PUF = 1;
			$display("%t WARINNG: POWER-UP Operation not completely performed. DO AGAIN A POWER ON RESET",$realtime);

		if((W !== 1'b0)&&(W !== 1'b1)&&(power_on == 1'b1)&& safe_reg[6] == 0)//&&(Qout_en == 0))
			$display("%t: WARNING: /W [%b] input is not driven or fast quad read instruction is coming, please don't let /W input pin unconnected!\n",$realtime,W);
	end

//==========================================
//Device Power On/Off
//==========================================
	always@(VCC)
	begin
		if(VCC == 1'b1)
		begin
			mode         = device_standby_mode;
			power_on     = 1'b1;
			power_off    = 1'b0;
			power_on_rst = 1'b1;

			$display("%t: NOTE: DEVICE IS POWERED ON!\n",$realtime);
		end
//-----------------------------------------------
		else begin
			if(power_on == 1'b1) $display("%t: NOTE: DEVICE IS POWERED OFF!\n",$realtime);
			mode         = device_no_power_mode;
			power_on     = 1'b0;
			power_off    = 1'b1;
		end
	end

//==========================================
//Hold Condition
//==========================================
	always@(HOLD)
	begin
		if(power_on == 1'b1 && safe_reg[6] == 0) //for PUF = 0
		begin
			if(HOLD == 1'b0 && Qout_en == 0)
			begin
				if(S == 1'b0)
				begin
					if(C == 1'b0)
					begin
						if(hold_stop_next == 1'b1) hold_stop_next = 1'b0;
						if(hold_condition == 1'b1)
							$display("%t: WARNING: This falling edge on HOLD has no effect because HOLD CONDITION has already started!",$realtime);
						else
						begin
							$display("%t: NOTE: COMMUNICATION PAUSED!",$realtime);
							clk_in = 1'b0; hold_condition = 1'b1;
						end
					end
					else if(C == 1'b1)
					begin
						if(hold_stop_next == 1'b1)
						begin
							hold_start_next = 1'b0; hold_stop_next = 1'b0;
						end
						else if(hold_condition == 1'b0) hold_start_next = 1'b1;
					//$display("%t: WARNING: This falling edge on HOLD can not start HOLD CONDITION because clock is high!",$realtime);
					end
				end
`ifdef SHOW_ALL_MSG
				else $display("%t: ERROR: The Device is not selected! To start the HOLD condition, /S must be driven low!",$realtime);
`endif
			end
			else if(HOLD == 1'b1 && Qout_en == 0)
			begin
				if(C == 1'b0)
				begin
					if(hold_start_next == 1'b1) hold_start_next = 1'b0;
					if(hold_condition == 1'b0)
`ifdef SHOW_ALL_MSG
						$display("%t: WARNING: This rising edge on HOLD has no effect because HOLD CONDITION has not started!",$realtime);
`else		
						;
`endif
					else
					begin
						$display("%t: NOTE: COMMUNICATION RESUME!\n",$realtime);
						clk_in = C; hold_condition = 1'b0;
					end
				end
				else if(C == 1'b1)
				begin
					if(hold_start_next == 1'b1)
					begin
						hold_start_next = 1'b0; hold_stop_next = 1'b0;
					end
					else if(hold_condition == 1'b1) hold_stop_next = 1'b1;
				//$display("%t: WARNING: This rising edge on HOLD can not end HOLD CONDITION because clock is high!",$realtime);
				end
			end
			else if (Qout_en == 1)
			begin
			//          $display("%t Fast Quad Read is running, then HOLD is an output", $realtime);
			end
			else $display("%t: WARNING: /HOLD [%b] input is not driven, please don't let /HOLD input pin unconnected!\n",$realtime,HOLD);
		end

		else  if(power_on == 1'b1 && safe_reg[6] == 1)  //for PUF = 1
			$display("%t WARINNG: POWER-UP Operation not completely performed. DO AGAIN A POWER ON RESET",$realtime);

		else $display("%t: ERROR: Device is not Powered on!\n",$realtime);
	end

//=========================================================
	always@(C)
	begin
		if(C == 1'b0)
		begin
			if(hold_start_next == 1'b1)
			begin
				hold_start_next = 1'b0; hold_condition = 1'b1;
				$display("%t: NOTE: COMMUNICATION PAUSED!",$realtime);
			//clk_in = 1'b0;
			end
			if(hold_stop_next == 1'b1)
			begin
				hold_stop_next = 1'b0; hold_condition = 1'b0;
				$display("%t: NOTE: COMMUNICATION RESUME!\n",$realtime);
			//clk_in = C;
			end
		end
		if(hold_condition == 1'b0) clk_in = C;
		if(hold_condition == 1'b1) clk_in = 1'b0;
	end

//=========================================================
//during HOLD Condition period, Serial clock is not be care
	always@(posedge hold_condition)
	begin
		Q = #(M95XXX_SIM.M95XXX_Macro_mux.tHLQZ) 1'bz;
	end

//---------------------------
	always@(negedge hold_condition)
	begin
		Q = #(M95XXX_SIM.M95XXX_Macro_mux.tHHQV) dout;
	end

//---------------------------
	always@(dout || S)
	begin
		if(hold_condition == 1'b0) Q = dout;
	end

	always@(posedge S)
	begin
		#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);
		Q = 1'bz;
		dout = 1'bz;
		DQ2 = 1'bz;

	end


//=========================================================
//chip select driven low, device active.
//=========================================================
	always@(negedge S)
	begin
		if(power_on == 1'b1 && safe_reg[6] == 0) //for PUF = 0
		begin
			if(W == 1'b0) wr_protect = 1'b1; //W=0 when chip select, Write Protect
			if((W !== 1'b0)&&(W !== 1'b1))
				$display("%t: WARNING: /W [%b] input is not driven or a fast quad read instruction is coming, please don't let /W input pin unconnected!\n",$realtime,W);
			if((HOLD !== 1'b0)&&(HOLD !== 1'b1))
				$display("%t: WARNING: /HOLD [%b] input is not driven, please don't let /HOLD input pin unconnected!\n",$realtime,HOLD);
			select_ok        = 1'b1;
			bit_counter_en   = 1'b1;
			bit_counter_ld   = 1'b1;
			instruction_byte = 1'b1;
			mode = active_power_mode;
		end
		else if(power_on == 1'b1 && safe_reg[6] == 1) //for PUF = 1
			$display("%t WARINNG: POWER-UP Operation not completely performed. DO AGAIN A POWER ON RESET",$realtime);
		else
			$display("%t: ERROR: Device is not Powered on!\n",$realtime);

	end

//=========================================================
//Serial data bit(on the "D" line) input
//=========================================================
	always@(posedge clk_in)
	begin
		if(power_on == 1'b1)
		begin
			if(S == 1'b0)
			begin
				//$display("%t MEMORY DEBUG sono in clk_in: bit_counter_en = %b bit_counter_ld = %b bit_counter = %b data_byte = %b byte_ok =%b   ",$realtime,bit_counter_en,bit_counter_ld,bit_counter, data_byte,byte_ok);
				//$display("%t MEMORY DEBUG mode = %b data_byte= %b byte_ok = %b", $realtime, mode, data_byte,byte_ok);
				if((bit_counter_en == 1'b1)&&(bit_counter_ld == 1'b1))
				begin

					bit_counter = 3'b111;
					shift_in_reg = {shift_in_reg[6:0],D};

					if(operation == WRITE_DATA_IN) write_data_byte_ok = 1'b0;
					if(operation == PGPR) begin
						write_data_byte_ok = 1'b0;
					end
					if(operation == PGPR_BUFF) begin
						write_data_byte_ok = 1'b0;
					end
					if(operation == WRITE_ID_DATA_IN) write_data_byte_ok = 1'b0;
					if(operation == EN_WRITE) write_en_id = 1'b0;
					if(operation == DIS_WRITE) write_dis_id = 1'b0;
					//if(operation == WRITE_SR_IN) write_new_sr_ok = 1'b0;
					if(operation == LOCKID) set_new_lockid_ok = 1'b0;
					if(operation == JEDEC_ID) set_new_jedec_ok = 1'b0;
					if(operation == SAFE_CLEAR_FLAG) clear_safe_reg_en = 1'b0;
					if(operation == ER_SECT) set_new_sect_eras_ok = 1'b0;
					if(operation == WRVR)
					begin

						vol_reg_en = 1'b0;

					end

					/////------- BLOCK ERASE INSTRUCTION
					if(operation == BKER) set_new_block_eras_ok = 1'b0;
					/////--------CHIP ERASE INSTRUCTION
					if(operation == CHER)chip_erase_en = 1'b0;
					////-------- RESET ENABLE & SOFTWARE RESET
					if(operation == RSTEN) reset_en_id = 1'b0;
					if(operation == RESET) res_soft_en = 1'b0;
					/////------- PAGE ERASE INSTRUCTION
					if(operation == PGER) set_new_page_eras_ok = 1'b0;
					////-------- DEEP POWER DOWN
					if(operation == DPD) dpd_en = 1'b0;
					////-------- RELEASE DEEP POWER DOWN
					if(operation == RDPD) rdpd_en = 1'b0;
				end
				else if((bit_counter_en == 1'b1)&&(bit_counter_ld == 1'b0))
				begin

					bit_counter = bit_counter - 3'b001;
					shift_in_reg = {shift_in_reg[6:0],D};
				end
				if((bit_counter_en == 1'b1)&&(bit_counter == 3'b000))
				begin

					byte_ok = 1'b1;
					bit_counter_en = 1'b0;

				end
				else if(bit_counter_en == 1'b1) bit_counter_ld = 1'b0;
			end
`ifdef SHOW_ALL_MSG
			else $display("%t: WARNING: Device is in standby mode! Falling edge on /S is required!\n",$realtime);
`endif
		end
		else $display("%t: ERROR: Device is not Powered on!\n",$realtime);
	end

/////////=============================== WRITE STATUS AND CONFIG REGISTERS=======================================////////////////////
	always@(posedge clk_in)
		if(operation == WRITE_SR_IN)
		begin
			if(power_on == 1'b1)
			begin
				if(S == 1'b0)
				begin
					if((bit_counter_en_wr == 1'b1)&&(bit_counter_ld_wr == 1'b1))
					begin
						write_new_sr_ok = 1'b0;
						bit_counter_wr = 4'b1111;
						shift_in_reg_new = {shift_in_reg_new[14:0],D};

					end
					else if((bit_counter_en_wr == 1'b1)&&(bit_counter_ld_wr == 1'b0))
					begin
						bit_counter_wr = bit_counter_wr - 4'b0001;
						shift_in_reg_new = {shift_in_reg_new[14:0],D};

						if(bit_counter_wr<=4'b1111 && bit_counter_wr >= 4'b1000)
						begin
							shift_in_reg = shift_in_reg_new[7:0];
							shift_config[7:0] = config_reg[7:0];

							byte_ok_wr = 1'b1;
						//$display("%t : MEMORY DEBUG : shift_in_reg=[%b]", $realtime, shift_in_reg);
						end
					end

					if((bit_counter_en_wr == 1'b1))
						if(bit_counter_wr == 3'b000)
						begin
							shift_in_reg = shift_in_reg_new[15:8];
							shift_config = shift_in_reg_new[7:0];
							byte_ok_wr = 1'b1;
							bit_counter_en_wr = 1'b0;
						//$display("%t : MEMORY DEBUG : shift_in_reg=[%b]", $realtime, shift_in_reg);
						//$display("%t : MEMORY DEBUG: shift_config=[%b]", $realtime, shift_config);
						end
						else if(bit_counter_en_wr == 1'b1) bit_counter_ld_wr = 1'b0;
				end
`ifdef SHOW_ALL_MSG
				else $display("%t: WARNING: Device is in standby mode! Falling edge on /S is required!\n",$realtime);
`endif
			end
			else $display("%t: ERROR: Device is not Powered on!\n",$realtime);

		end
//////////=================================================================///////////////////


//===================================================================
//chip select driven high, internal write cycle and standby
//===================================================================
	//----------------------------Read status register

	always@(posedge S)              //chip select driven high,device disable.
	begin : evento_posedgeS
				
		if(operation == READ_SR_OUT)
		begin
			operation = 5'b00000;
			instruction = 5'b0000;
			$display("%t: NOTE: Read status register operation is finished.\n",$realtime);
			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);
			rds_inprogress = 1'b0;
		end

		if(operation == JEDEC_ID)
		begin
			operation = 5'b00000;
			instruction = 5'b00000;
			$display("%t MEMORY : ST_manufacturer_code = %b | SPI_Family_code = %b | Memory_density = %b",$realtime,memory_id[0], memory_id[1],memory_id[2]);
			$display("%t MEMORY : out_jedec = %b = %h",$realtime, out_jedec,out_jedec);
			$display("%t: NOTE: JEDEC ID read operation is finished.\n",$realtime);
			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);
			jedid_inprogress = 1'b0;
		end

/////================ READ VOLATILE REGISTER INSTRUCTION ===========/////
		if(operation == RDVR)
		begin
			operation = 5'b00000;
			instruction = 5'b00000;
			$display("%t: NOTE: Read Volatile register operation is finished.\n",$realtime);
			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);
			rdv_inprogress = 1'b0;
		end
/////===============================================================/////

		if(operation == RDCR)
		begin
			operation = 5'b00000;
			instruction = 5'b00000;

			if(safe_reg != 8'h00)
			begin
				if (safe_reg[7]) $display("%t WARNING: A modify operation has been attempted to a protected memory area! \n \t \t \t \t \t PAMAF bit in safety register is %b  ",$realtime, safe_reg[7]);
				if (safe_reg[6]) $display("%t WARNING: Power Up operation wasn't completed successfully! \n \t \t \t \t \t PUF bit in safety register is %b ",$realtime, safe_reg[6]);
				if (safe_reg[5]) $display("%t WARNING: Last erase/write operation wasn't completed successfully! \n \t \t \t \t \t ERF bit in safety register is %b ",$realtime, safe_reg[5]);
				if (safe_reg[4]) $display("%t WARNING: Last program/write operation wasn't completed successfully! \n \t \t \t \t \t  PRF bit in safety register is %b ",$realtime, safe_reg[4]);
				if (safe_reg[3]) $display("%t WARNING: In last executed operation ECC has corrected a bit error \n \t \t \t \t \t ECC1C bit in safety register is %b ",$realtime, safe_reg[3]);
				if (safe_reg[2]) $display("%t WARNING: In last executed operation ECC has corrected a double bit error \n \t \t \t \t \t ECC2C bit in safety register is %b ",$realtime, safe_reg[2]);
				if (safe_reg[1]) $display("%t WARNING: In last executed operation ECC has detected a triple bit error in at least one word of 16 bytes \n \t \t \t \t \t ECC3D bit in safety register is %b ",$realtime, safe_reg[1]);
				if (safe_reg[0]) $display("%t WARNING: In last executed operation ECC has detected a triple bit error in at least one word of 16 bytes \n \t \t \t \t \t ECC3DS bit in safety register is %b ",$realtime, safe_reg[0]);
			end

			$display("%t MEMORY : Configuration Register = %b | Safety Register = %b | ",$realtime,config_reg, safe_reg,);
			$display("%t MEMORY : out_conf_safe_reg = %b [%h h]",$realtime, out_conf_safe_reg,out_conf_safe_reg);
			$display("%t: NOTE: READ CONFIGURATION AND SAFETY REGISTER operation is finished.\n",$realtime);
			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);
			read_config_safe_inprogress = 1'b0;
		end
	end //dell always@(posedge S)

	always @(posedge S)
	begin
		
		//---------------------------- Enable Reset Instruction Execute
		if(operation == RSTEN)
		begin
			operation = 5'b00000;
			//  $display(" reset en id %b  reset_en_bit %b",reset_en_id,reset_en_bit);
			if(reset_en_id == 1'b1)
			begin
				reset_en_id   = 1'b0;
				$display("%t: NOTE: ENABLE RESET.\n",$realtime);
				$display("%t: WARNING: ENABLE RESET INSTRUCTION is starting!",$realtime);
				reset_en_bit = 1'b1;
			end
			else if(reset_en_id == 1'b0)
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of the enable reset instruction code is latched. Enable Reset instruction is not executed.\n",$realtime);

		end

		/////====================================== Software Reset instruction =======/////
		if(operation == RESET)
		begin
			operation = 5'b00000;
			t_wait_reset =  $realtime();
			if(reset_en_bit == 1'b0)
			begin
				$display("%t ERROR: The SOFTWARE RESET instruction cannot be accepted because the Reset Enable instruction is not executed!\n", $realtime);

			end
			else
			begin
				if(res_soft_en == 1'b1)
				begin
					if(status_reg[0] == 1'b1) //operation in corso
					begin
						if(modify_operation)// This bit is used to identify if the operation driven before the RESET instruction is a MODIFY OPERATION (WRITE) or not!!! This doesn't include the CHER instruction
						begin
							#(M95XXX_SIM.M95XXX_Macro_mux.tRST2-(t_wait_reset-t_modify));   //// Reset Time when Reset occurs in modify operation except CHER instruction
							res_soft_en = 1'b0;
							reset_en_bit = 1'b0;
							power_on_rst = 1'b1;
							//mode = device_standby_mode;
							//$display("%t MEMORY DEBUG : sono nel caso tRST2",$realtime);
							//$display("%t  DEBUG: volatile_reg =[%b]",$realtime, volatile_reg);
							//$display("%t  DEBUG: mode1 = %b",$realtime,mode);
							//$display("%t  DEBUG: power_on_reset is [%b]",$realtime,power_on_rst);
							//$display("%t  DEBUG: reset_en_bit is [%b]",$realtime,reset_en_bit);
							//$display("%t  DEBUG: res_soft_en is [%b]",$realtime,res_soft_en);
							$display("%t: NOTE: SOFTWARE RESET.\n",$realtime);
							$display("%t: NOTE: Begin to RESET instruction",$realtime);
						end
						else if (modify_operation_cher)
						begin
							#(M95XXX_SIM.M95XXX_Macro_mux.tRST3-(t_wait_reset - t_chip_erase)); //// Reset Time when Reset occurs in chip erase execution
							res_soft_en = 1'b0;
							reset_en_bit = 1'b0;
							power_on_rst = 1'b1;
							//$display("%t MEMORY DEBUG : sono nel caso tRST3",$realtime);
							//$display("%t DEBUG : power_on_reset is [%b]",$realtime,power_on_rst); //SOLO PER AVERE CONFERMA PERCHE' SIMVISION NON AGGIORNA IL BIT
							//$display("%t DEBUG : reset_en_bit is [%b]",$realtime,reset_en_bit);  //SOLO PER AVERE CONFERMA PERCHE' SIMVISION NON AGGIORNA IL BIT
							//$display("%t DEBUG : res_soft_en is [%b]",$realtime,res_soft_en);   //SOLO PER AVERE CONFERMA PERCHE' SIMVISION NON AGGIORNA IL BIT
							//$display("%t DEBUG : WIP = %b", $realtime, status_reg[0]);
							$display("%t: NOTE: SOFTWARE RESET.\n",$realtime);
							$display("%t: NOTE: Begin to RESET instruction",$realtime);
						end


					end

				end

				else
					$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of data is latched, RESET instruction instruction is not executed.\n",$realtime);

			end
		end
	/////======================================================================/////

	end

	always@(posedge S)              //chip select driven high,device disable.
	begin : attivazione_pgpr
				
		select_ok = 1'b0;

		start_PGPR = 1'b0;
		//$display("%t DEBUG: sono nel posedge S del task.  mode = %b  byteok =%b  new_volatile_byte = %b",$realtime,mode,byte_ok,new_volatile_byte);
		//$display("%t DEBUG: operation %b reset_en_id %b reset_en_bit %b ",$realtime, operation, reset_en_id,reset_en_bit);


		//used on last bit of last data byte when the S enable is deactivated when byte_ok is still set last falling edge clk never arrives
		if((mode == active_power_mode)&&(data_byte == 1'b1)&&(byte_ok == 1'b1))
		begin
			
			byte_ok = 1'b0;
			data_latch[page_offset_addr] = shift_in_reg;
			if(page_offset_addr == {`PAGE_OFFSET_BITS {1'b1}}) page_offset_addr = {`PAGE_OFFSET_BITS {1'b0}};
			else page_offset_addr = page_offset_addr + {{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
			end_address = page_offset_addr;
			bit_counter_en = 1'b1;
			bit_counter_ld = 1'b1;
			write_data_byte_ok = 1'b1;
		end


////////------- ------------ FOR SECTOR ERASE INSTRUCTION
		if((mode == active_power_mode)&&(data_byte_sect == 1'b1)&&(byte_ok == 1'b1))
		begin

			byte_ok = 1'b0;
			data_latch[page_offset_addr] = shift_in_reg;
			bit_counter_en = 1'b1;
			bit_counter_ld = 1'b1;
			set_new_sect_eras_ok = 1'b1;
		end

		///////---------------------FOR BLOCK ERASE INSTRUCTION
		if((mode == active_power_mode)&&(data_byte_block == 1'b1)&&(byte_ok == 1'b1))
		begin

			byte_ok = 1'b0;
			data_latch[page_offset_addr] = shift_in_reg;
			bit_counter_en = 1'b1;
			bit_counter_ld = 1'b1;
			set_new_block_eras_ok = 1'b1;
		end

		////////------- ------------ FOR PAGE ERASE INSTRUCTION
		if((mode == active_power_mode)&&(data_byte_page == 1'b1)&&(byte_ok == 1'b1))
		begin
			
			byte_ok = 1'b0;
			data_latch[page_offset_addr] = shift_in_reg;
			bit_counter_en = 1'b1;
			bit_counter_ld = 1'b1;
			set_new_page_eras_ok = 1'b1;
		end

		//----------------------------new SR value input --used on last bit of last data byte when the S enable is deactivated when byte_ok is still set last falling edge clk never arrives
		if((mode == active_power_mode)&&(new_sr_byte == 1'b1)&&(byte_ok == 1'b1 || byte_ok_wr == 1'b1))
		begin

			byte_ok_wr = 1'b0;
			byte_ok = 1'b0;
			sr_latch = shift_in_reg;
			config_latch_new = shift_config;
			new_sr_byte = 1'b0;
			write_new_sr_ok = 1'b1;
		end

/////========================= FOR WRITE VOLATILE REGISTER =======/////
		if((mode == active_power_mode)&&(new_volatile_byte == 1'b1)&&(byte_ok == 1'b1))
		begin
			
			byte_ok = 1'b0;
			volatile_latch = shift_in_reg;
			vol_reg_en = 1'b1;
			new_volatile_byte = 1'b0;
		end

/////================================================================/////


		//---------------------------- Write Lock ID --used on last bit of last data byte when the S enable is deactivated when byte_ok is still set last falling edge clk never arrives
		if((mode == active_power_mode)&&(lock_byte == 1'b1)&&(byte_ok == 1'b1))
		begin

			byte_ok = 1'b0;
			lock_latch = shift_in_reg;
			lock_byte = 1'b0;
			set_new_lockid_ok = 1'b1;
		end

		//----------------------------Read Data From Memory Array
		if(operation == READ_DATA_OUT)
		begin
			//$display(" DEBUG: DATA READING START");
			if(i==0)
			begin
				if(memory_address[`MEM_ADDR_BITS-1:0] == {`MEM_ADDR_BITS {1'b1}}) //`MEM_ADDR_BITS'h3ff)
					memory_address[`MEM_ADDR_BITS-1:0] = {`MEM_ADDR_BITS {1'b0}};
				else memory_address[`MEM_ADDR_BITS-1:0] = memory_address[`MEM_ADDR_BITS-1:0] + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};// `MEM_ADDR_BITS'h001;
				i = 8;
			end
			data_out_buf = memory[memory_address[`MEM_ADDR_BITS-1:0]];
			dout = data_out_buf[i-1];
			i = i-1;

		end

		//----------------------------Fast Single Read Data From Memory Array
		if(operation == FREAD)
		begin
			if(i==0)
			begin

				if(memory_address[`MEM_ADDR_BITS-1:0] == {`MEM_ADDR_BITS {1'b1}}) //`MEM_ADDR_BITS'h3ff)
					memory_address[`MEM_ADDR_BITS-1:0] = {`MEM_ADDR_BITS {1'b0}};
				else memory_address[`MEM_ADDR_BITS-1:0] = memory_address[`MEM_ADDR_BITS-1:0] + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};// `MEM_ADDR_BITS'h001;
				i = 8;
			end
			data_out_buf = memory[memory_address[`MEM_ADDR_BITS-1:0]];
			dout = data_out_buf[i-1];
			//$display("%t MEMORY DEBUG:  data_out_buf = %h  memory_add = %h  i = %d ",$realtime,data_out_buf,memory_address,i);
			i = i-1;
		end
		//----------------------------Fast Dual Read Data From Memory Array
		if(operation == FDREAD)
		begin

			if(i==0)
			begin

				if(memory_address[`MEM_ADDR_BITS-1:0] == {`MEM_ADDR_BITS {1'b1}}) //`MEM_ADDR_BITS'h3ff)
					memory_address[`MEM_ADDR_BITS-1:0] = {`MEM_ADDR_BITS {1'b0}};
				else memory_address[`MEM_ADDR_BITS-1:0] = memory_address[`MEM_ADDR_BITS-1:0] + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};// `MEM_ADDR_BITS'h001;
				i = 4;
			end
			data_out_buf = memory[memory_address[`MEM_ADDR_BITS-1:0]];
			DQ0 = data_out_buf[2*i];
			dout = data_out_buf[2*i+1];
			//$display("%t MEMORY DEBUG:  data_out_buf = %h  memory_add = %h  i = %d ",$realtime,data_out_buf,memory_address,i);
			i = i-1;
		end

		//----------------------------Fast Quad Read Data From Memory Array
		if(operation == FQREAD)
		begin
			if(i==0)
			begin

				if(memory_address[`MEM_ADDR_BITS-1:0] == {`MEM_ADDR_BITS {1'b1}}) //`MEM_ADDR_BITS'h3ff)
					memory_address[`MEM_ADDR_BITS-1:0] = {`MEM_ADDR_BITS {1'b0}};
				else memory_address[`MEM_ADDR_BITS-1:0] = memory_address[`MEM_ADDR_BITS-1:0] + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};// `MEM_ADDR_BITS'h001;
				i = 2;
			end
			data_out_buf = memory[memory_address[`MEM_ADDR_BITS-1:0]];
			DQ0 = data_out_buf[4*i];
			dout = data_out_buf[4*i+1];
			DQ2 = data_out_buf[4*i+2];
			DQ3 = data_out_buf[4*i+3];
			i = i-1;
		end
		//----------------------------Read Data From ID Memory Array or Read lock status

		if(operation == READ_ID_DATA_OUT)
		begin
			if(i==0)
			begin
				if((readidwrap)&&(memory_address[`PAGE_OFFSET_BITS:0] == {{`PAGE_OFFSET_BITS {1'b0}},1'b1})) $display("%t: ERROR:  The number of bytes read has exceeded ID page boundary [%d], unexpected data is read\n",$realtime,page_size);

				if(memory_address[`PAGE_OFFSET_BITS:0] == {{`PAGE_OFFSET_BITS {1'b1}},1'b1})
				begin
					memory_address[`PAGE_OFFSET_BITS:0] = {{`PAGE_OFFSET_BITS {1'b0}},1'b0};
					$display("%t: NOTE:  The number of bytes read has reached ID page boundary [%d]\n",$realtime,2*page_size-1);
					readidwrap = 1'b1;
				end
				else memory_address[`PAGE_OFFSET_BITS:0] = memory_address[`PAGE_OFFSET_BITS:0] + {{`PAGE_OFFSET_BITS {1'b0}},1'b1};
				i = 8;
			end
			data_out_buf = memory_id[memory_address[`PAGE_OFFSET_BITS:0]];
			dout = data_out_buf[i-1];
			i = i-1;
		end

//=================== FOR FAST READ IDENTIFICATION PAGE ===========//
		//----------------------------Read Data From ID Memory Array or Read lock status

		if(operation == FRDID)
		begin
			if(i==0)
			begin
				if(memory_address[`PAGE_OFFSET_BITS:0] == {{`PAGE_OFFSET_BITS {1'b1}},1'b1})
				begin
					memory_address[`PAGE_OFFSET_BITS:0] = {{`PAGE_OFFSET_BITS {1'b0}},1'b0};
					$display("%t: NOTE:  The number of bytes read has reached ID page boundary [%d]\n",$realtime,2*page_size-1);
				end
				else memory_address[`PAGE_OFFSET_BITS:0] = memory_address[`PAGE_OFFSET_BITS:0] + {{`PAGE_OFFSET_BITS {1'b0}},1'b1};
				i = 8;
			end
			data_out_buf = memory_id[memory_address[`PAGE_OFFSET_BITS:0]];
			dout = data_out_buf[i-1];
			i = i-1;
		end

//===================================================================//
//=================== FOR READ SFDP_Register  ===========//
		//----------------------------Read Data From SFDP_Register

		if(operation == RDSFDP)
		begin
			if(i==0)
			begin

				if(memory_address[`PAGE_OFFSET_BITS-1:0] == {`PAGE_OFFSET_BITS {1'b1}})
				begin
					memory_address[`PAGE_OFFSET_BITS-1:0] = {`PAGE_OFFSET_BITS {1'b0}};
					$display("%t: NOTE:  The number of bytes read has reached SFDP_Register boundary [%d]\n",$realtime,page_size-1);

				end
				else memory_address[`PAGE_OFFSET_BITS-1:0] = memory_address[`PAGE_OFFSET_BITS-1:0] + {{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
				i = 8;
			end
			data_out_buf = sfdp_register[memory_address[`PAGE_OFFSET_BITS-1:0]];
			dout = data_out_buf[i-1];
			i = i-1;
		end


		//----------------------------incorrect /S input checking
		if(instruction_byte == 1'b1 && !byte_ok) $display("%t: ERROR: /S is driven high during the instruction byte latched in.",$realtime);
		if(address_h2_byte  == 1'b1) $display("%t: ERROR: /S is driven high during the high byte 2 address byte latched in.",$realtime);
		if(address_h_byte   == 1'b1) $display("%t: ERROR: /S is driven high during the high byte 1 address byte latched in.",$realtime);
		if(address_l_byte   == 1'b1) $display("%t: ERROR: /S is driven high during the low address byte latched in.",$realtime);
		if(hold_condition   == 1'b1)
		begin
			hold_condition = 1'b0;
			$display("%t: WARNING: /S is driven high when memory is in the HOLD Condition, Reset the current state of memory!",$realtime);
		end
		//----------------------------Write Enable Instruction Execute
		if(operation == EN_WRITE)
		begin
			operation = 5'b00000;
			modify_operation =1'b0;
			modify_operation_cher =1'b0;
			if(write_en_id == 1'b1)
			begin
				write_en_id   = 1'b0;
				status_reg[1] = 1'b1;     //WEL is Set
				$display("%t: NOTE: WRITE ENABLE.\n",$realtime);
			end
			else if(write_en_id == 1'b0)
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of the write enable instruction code is latched. Write Enable instruction is not executed.\n",$realtime);
		end
		//----------------------------Write Disable Instruction Execute
		if(operation == DIS_WRITE)
		begin
			operation = 5'b0000;
			modify_operation =1'b0;
			modify_operation_cher =1'b0;
			if(write_dis_id == 1'b1)
			begin
				write_dis_id  = 1'b0;
				status_reg[1] = 1'b0;     //WEL is Reset
				$display("%t: NOTE: WRITE DISABLE.\n",$realtime);
			end
			else if(write_dis_id == 1'b0)
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of the write disable instruction code is latched. Write Disable instruction is not executed.\n",$realtime);
		end

		//----------------------------Write SR Instruction Execute
		if(operation == WRITE_SR_IN)
		begin
			t_modify = $realtime;
			operation = 5'b00000;
			modify_operation =1'b1;
			modify_operation_cher =1'b0;
			if(write_new_sr_ok == 1'b1)
			begin
				write_new_sr_ok = 1'b0;
				if((wr_protect == 1'b1)&&(status_reg[7] == 1'b1)) //Hardware write protect
				begin
					$display("%t  WARNING: The Status Register is in read-only mode, the WRSR instruction will not be executed.", $realtime);
					$display("%t  Hardware Write Protected Mode(HPM) is entered! | SRWD = %b | W = %b |",$realtime, status_reg[7],W);
				end
				else
				begin
					$display("%t: NOTE: Begin to Write Status Register with data %b!",$realtime,sr_latch);
					status_reg[0] = 1'b1;           //WIP is 1 during this cycle
					#(M95XXX_SIM.M95XXX_Macro_mux.tWSCR);                         //Write time
					status_reg[4:2]= sr_latch[4:2];//BP2 //BP1,BP0
					status_reg[7:6] = sr_latch[7:6];    //SRWD  //TB
					config_reg[7:0] = config_latch_new[7:0];
					$display("%t: NOTE: Write Status Register successfully! (WROTE %b but actually is sr = [%b]",$realtime,sr_latch,status_reg);
					status_reg[0] = 1'b0;           //WIP is reset when write cycle is completed.
					status_reg[1] = 1'b0;           //WEL is reset when write cycle is completed.
				end
			end
			else if(write_new_sr_ok == 1'b0)
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of data is latched, Write SR instruction is not executed.\n",$realtime);

			nonxt_byte = 1'b0;
		end

		/////====================================== WRITE VOLATILE REGISTER =======/////
		if(operation == WRVR)
		begin
			operation = 5'b00000;
			modify_operation =1'b1;
			modify_operation_cher =1'b0;

			//$display("%t MEMORY: vol_reg_en = %b",$realtime, vol_reg_en);
			if(vol_reg_en == 1'b1)
			begin
				vol_reg_en = 1'b0;
				$display("%t: NOTE: Begin to Write Volatile Register!",$realtime);
				status_reg[0] = 1'b1;           //WIP is 1 during this cycle

				//added for page program
				page_prog_buff_mode_start = 1'b0;



				volatile_reg[1] = volatile_latch[1];
				volatile_reg[0] = ~volatile_reg[1];
				$display("%t: NOTE : Write Volatile Register successfully! Now it is %b ",$realtime, volatile_reg);
				$display("%t: NOTE : BUFLD bit is set to %d because it is the negate of BUFEN\n",$realtime, volatile_reg[0]);
				$display("%t: NOTE : BUFEN bit is set to %d", $realtime, volatile_reg[1]);
				status_reg[0] = 1'b0;           //WIP is reset when write cycle is completed.
				status_reg[1] = 1'b0;           //WEL is reset when write cycle is completed.
				for(i=2;i<=7;i=i+1)
				begin
					$display("%t: ERROR : Bit[%d] can not be written but it is accessed in only-read mode\n",$realtime,i);
				end
				i=2;
			end
			else
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of data is latched, Write Volatile instruction is not executed.\n",$realtime);
		end
		/////======================================================================/////

//---------------------------- Deep Power Down Instruction Execute
		if(operation == DPD)
		begin
			operation = 5'b00000;
			if(dpd_en == 1'b1)
			begin
				dpd_en   = 1'b0;

				$display("%t: NOTE: DEEP POWER DOWN \n",$realtime);
				$display("%t: WARNING: DEEP POWER DOWN INSTRUCTION is starting!",$realtime);
				#(M95XXX_SIM.M95XXX_Macro_mux.tDP);
				// $display("%t : I'm entering in DPD", $realtime);
				mode = deep_power_mode; //11
				deep_pwr_active = 1'b1;
				//Icc2= 1'b0;
				// $display("%t : mode = %b",$realtime, mode);
				disable attivazione_pgpr;
			end
			else if(dpd_en == 1'b0)
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of the enable reset instruction code is latched. Deep Power Down instruction is not executed.\n",$realtime);


		end

//---------------------------- Release Deep Power Down Instruction Execute
		if(operation == RDPD)
		begin
			operation = 5'b00000;
			if(rdpd_en == 1'b1)
			begin
				rdpd_en   = 1'b0;

				$display("%t: NOTE: RELEASE DEEP POWER DOWN \n",$realtime);
				$display("%t: WARNING: RELEASE DEEP POWER DOWN INSTRUCTION is starting!",$realtime);
				#(M95XXX_SIM.M95XXX_Macro_mux.tRDPD);
				//$display("%t : I'm entering in RDPD", $realtime);
				mode = device_standby_mode; //10
				//   Icc2= 1'b0;
				//  $display("%t : mode = %b",$realtime, mode);
				#(M95XXX_SIM.M95XXX_Macro_mux.tRDPSL);
				disable attivazione_pgpr;
			end
			else if(rdpd_en == 1'b0)
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of the enable reset instruction code is latched. Release Deep Power Down instruction is not executed.\n",$realtime);


		//$display("%t : 2mode = %b",$realtime, mode);

		end


//---------------------------- Enable Reset Instruction Execute
		if(operation == RSTEN)
		begin
			operation = 5'b00000;
			// $display(" reset en id %b  reset_en_bit %b",reset_en_id,reset_en_bit);
			if(reset_en_id == 1'b1)
			begin
				reset_en_id   = 1'b0;
				$display("%t: NOTE: ENABLE RESET.\n",$realtime);
				$display("%t: WARNING: ENABLE RESET INSTRUCTION is starting!",$realtime);
				reset_en_bit = 1'b1;
			end
			else if(reset_en_id == 1'b0)
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of the enable reset instruction code is latched. Enable Reset instruction is not executed.\n",$realtime);

		end

		/////====================================== Software Reset instruction =======/////
		if(operation == RESET)
		begin
			operation = 5'b00000;
			t_wait_reset =  $realtime();
			if(reset_en_bit == 1'b0)
			begin
				$display("%t ERROR: The SOFTWARE RESET instruction cannot be accepted because the Reset Enable instruction is not executed!\n", $realtime);

			end

			else
			begin
				if(res_soft_en == 1'b1)
				begin
					if(mode == deep_power_mode) //if(mode == device_standby_mode || mode == deep_power_mode) //if(status_reg[0] == 1'b0 || mode == deep_power_mode)  ///// WIP=0 oppure il device e' in deep power mode)
					begin
						#(M95XXX_SIM.M95XXX_Macro_mux.tRST1);   //// Reset Time when Reset occurs with WIP=0
						res_soft_en = 1'b0;
						reset_en_bit = 1'b0;
						power_on_rst = 1'b1;

						mode = device_standby_mode; 
//                          $display("%t MEMORY DEBUG : sono nel caso tRST1",$realtime);
//                          $display("%t  DEBUG: mode = %b",$realtime,mode); 
//                          $display("%t  DEBUG: power_on_reset is [%b]",$realtime,power_on_rst); 
//                          $display("%t  DEBUG: reset_en_bit is [%b]",$realtime,reset_en_bit);  
//                          $display("%t  DEBUG: res_soft_en is [%b]",$realtime,res_soft_en);   
//                          $display("%t  DEBUG: WIP = %b", $realtime, status_reg[0]);
						$display("%t: NOTE: SOFTWARE RESET.\n",$realtime);
						$display("%t: NOTE: Begin to RESET instruction",$realtime);
					end

					else
						if(status_reg[0] == 1'b1)
						begin
							if(modify_operation)
							begin
								#(M95XXX_SIM.M95XXX_Macro_mux.tRST2-(t_wait_reset-t_modify));   //// Reset Time when Reset occurs in modify operation except CHER instruction
								res_soft_en = 1'b0;
								reset_en_bit = 1'b0;
								power_on_rst = 1'b1;
								//mode = device_standby_mode;
//                                      $display("%t MEMORY DEBUG: sono nel caso tRST2",$realtime);
//                                      $display("%t  DEBUG: volatile_reg =[%b]",$realtime, volatile_reg); 
//                                      $display("%t  DEBUG: mode1 = %b",$realtime,mode); 
//                                      $display("%t  DEBUG: power_on_reset is [%b]",$realtime,power_on_rst); 
//                                      $display("%t  DEBUG: reset_en_bit is [%b]",$realtime,reset_en_bit);  
//                                      $display("%t  DEBUG: res_soft_en is [%b]",$realtime,res_soft_en);   
								$display("%t: NOTE: SOFTWARE RESET.\n",$realtime);
								$display("%t: NOTE: Begin to RESET instruction",$realtime);
							end
							else if (modify_operation_cher)
							begin
								#(M95XXX_SIM.M95XXX_Macro_mux.tRST3-(t_wait_reset - t_chip_erase)); //// Reset Time when Reset occurs in chip erase execution
								res_soft_en = 1'b0;
								reset_en_bit = 1'b0;
								power_on_rst = 1'b1;

//                                          $display("%t MEMORY DEBUG : sono nel caso tRST3",$realtime);
//                                          $display("%t  DEBUG: power_on_reset is [%b]",$realtime,power_on_rst); 
//                                          $display("%t  DEBUG: reset_en_bit is [%b]",$realtime,reset_en_bit);  
//                                          $display("%t  DEBUG: res_soft_en is [%b]",$realtime,res_soft_en);   
//                                          $display("%t  DEBUG: WIP = %b", $realtime, status_reg[0]);
								$display("%t: NOTE: SOFTWARE RESET.\n",$realtime);
								$display("%t: NOTE: Begin to RESET instruction",$realtime);
							end

						end
						else if(status_reg[0] == 1'b0)
						begin

							#(M95XXX_SIM.M95XXX_Macro_mux.tRST1); //// Reset Time when Reset occurs with WIP=0
							res_soft_en = 1'b0;
							reset_en_bit = 1'b0;
							power_on_rst = 1'b1;
							mode = device_standby_mode;
//                                        $display("%t MEMORY DEBUG : sono nel caso tRST1",$realtime);
//                                        $display("%t DEBUG : mode = %b",$realtime,mode); 
//                                        $display("%t DEBUG : power_on_reset is [%b]",$realtime,power_on_rst); 
//                                        $display("%t DEBUG : reset_en_bit is [%b]",$realtime,reset_en_bit);  
//                                        $display("%t DEBUG : res_soft_en is [%b]",$realtime,res_soft_en);   
//                                        $display("%t DEBUG : WIP = %b", $realtime, status_reg[0]);
							$display("%t: NOTE: SOFTWARE RESET.\n",$realtime);
							$display("%t: NOTE: Begin to RESET instruction",$realtime);


						end
				end

				else
					$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of data is latched, RESET instruction instruction is not executed.\n",$realtime);

			end
		end
		/////======================================================================/////


		//----------------------------
		if(operation == READ_DATA_OUT)
		begin
			t_modify = $realtime;
			modify_operation =1'b0;
			modify_operation_cher =1'b0;
			operation = 5'b00000;
			$display("%t: NOTE: Read data bytes operation is finished.\n",$realtime);
			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);
		end
		//----------------------------
		if(operation == FREAD)
		begin
			t_modify = $realtime;
			operation = 5'b00000;
			modify_operation =1'b0;
			modify_operation_cher =1'b0;
			$display("%t: NOTE: Fast Single Read data bytes operation is finished.\n",$realtime);
			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);
		end
		//----------------------------
		if(operation == FDREAD)
		begin
			t_modify = $realtime;
			operation = 5'b00000;
			modify_operation =1'b0;
			modify_operation_cher =1'b0;
			Qout_en_dual = 1'b0;
			$display("%t: NOTE: Fast Dual Read data bytes operation is finished.\n",$realtime);
			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);

		end
		//----------------------------
		if(operation == FQREAD)
		begin
			t_modify = $realtime;
			operation = 5'b00000;
			modify_operation =1'b0;
			modify_operation_cher =1'b0;
			Qout_en = 1'b0;
			$display("%t: NOTE: Fast Quad Read data bytes operation is finished.\n",$realtime);
			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);

		end
		//----------------------------
		if(operation == READ_ID_DATA_OUT)
		begin
			t_modify = $realtime;
			operation = 5'b00000;
			modify_operation =1'b0;
			modify_operation_cher =1'b0;
			readidwrap = 1'b0;
			if(!config_reg[0]) $display("%t: NOTE: Read ID data bytes operation is finished.\n",$realtime);
			else $display("%t: NOTE: Read Lock Status data bytes operation is finished.\n",$realtime);
			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);

		end
		//-----------------------------------
		//================ FAST READ IDENTIFICATION PAGE ===========// 
		if(operation == FRDID)
		begin
			t_modify = $realtime;
			modify_operation =1'b0;
			modify_operation_cher =1'b0;
			operation = 5'b00000;
			readidwrap = 1'b0;
			if(!config_reg[0]) $display("%t: NOTE: Read ID data bytes operation is finished.\n",$realtime); 
			else $display("%t: NOTE: Read Lock Status data bytes operation is finished.\n",$realtime);
			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);

		end
//-----------------------------------

//================  READ SFDP register ===========//
		////////======================================////
		if(operation == RDSFDP)
		begin
			t_modify = $realtime;
			modify_operation =1'b0;
			modify_operation_cher =1'b0;
			operation = 5'b00000;

			$display("%t: NOTE: Read SFDP_Register data bytes operation is finished.\n",$realtime);

			#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);

		end

		//----------------------------Clear Safety Register (CLRSF) Instruction Execute
		if(operation == SAFE_CLEAR_FLAG)
		begin
			modify_operation =1'b1;
			modify_operation_cher =1'b0;
			operation = 5'b00000;
			if(clear_safe_reg_en == 1'b1)
			begin
				clear_safe_reg_en = 1'b0;
				safe_reg[7:0] = 8'b00000000;
				$display("%t: NOTE: CLEAR SAFETY REGISTER.\n",$realtime);
				$display("%t: MEMORY: the safety register is reset [%h]", $realtime, safe_reg);
			end
			else if(clear_safe_reg_en == 1'b0)
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of the write enable instruction code is latched. Clear Safety Register instruction is not executed.\n",$realtime);
		end

////==============================//////
//---------------------------- CHIP ERASE (CHER) Instruction Execute
		if(operation == CHER)
		begin
			t_chip_erase = $realtime;
			modify_operation_cher =1'b1;
			//$display("%t DEBUG : modify_operation_cher = %b",$realtime, modify_operation_cher);
			operation = 5'b00000;

			if(chip_erase_en == 1'b1)
			begin
				chip_erase_en = 1'b0;
				start_address_chip = {`MEM_ADDR_BITS{1'b0}};


				if(status_reg[4] || status_reg[3] || status_reg[2])
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE CHIP ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : The whole Memory or its portion is Write Protected in accordance with BP2 = %b, BP1 = %b, BP0 =%b\n", $realtime, status_reg[4], status_reg[3], status_reg[2]);
					$display("%t: ERROR : The CHIP ERASE instruction is not accepted!\n",$realtime);
				end

				else begin
					status_reg[0] = 1'b1;
					#(M95XXX_SIM.M95XXX_Macro_mux.tCE);

					while(start_address_chip <(`MEM_SIZE-1))
					begin
						memory[start_address_chip] = 'hFF;
						//$display("%t DEBUG: Memory content is %h at address %h",$realtime,memory[start_address_chip],start_address_chip);//TOGLIERE IL COMMENTO SE SI VOGLIONO VEDERE STAMPATI A VIDEO INDIRIZZO E CONTENUTO
						start_address_chip = start_address_chip + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};
					end

					status_reg[0] = 1'b0;           //WIP is 0 when this cycle completed
					status_reg[1] = 1'b0;          //WEL must be set to 0 after the ERASE cycle
					safe_reg[5] = 1'b0;            //ERF=0 if the CHIP ERASE instruction is verified successfully; otherwise ERF=1.

					memory[start_address_chip] = 'hFF;
					//$display("%t DEBUG: Memory content is %h at address %h",$realtime,memory[start_address_chip],start_address_chip);//TOGLIERE IL COMMENTO SE SI VOGLIONO VEDERE STAMPATI A VIDEO INDIRIZZO E CONTENUTO

					$display("%t: NOTE : The CHIP ERASE instruction is finished successfully!!.",$realtime);

				end


			end
			else if(chip_erase_en == 1'b0)
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of the write enable instruction code is latched. Clear Safety Register instruction is not executed.\n",$realtime);
		end

//-------------  SECTOR ERASE OPERATION
		if(operation == ER_SECT)
		begin
			t_modify = $realtime;
			modify_operation = 1'b1;
			modify_operation_cher =1'b0;
			sector_inprogress = 1'b1;
			data_byte_sect = 1'b0;
			operation = 5'b00000;
			page_current = page_address_sect;
			address_no = start_address_sect;
			$display("%t: WARNING : SECTOR ERASE INSTRUCTION is starting!\n",$realtime);
			$display("%t: NOTE : Inserted Address[%h] is located in PAGE[%d]\n",$realtime,address_no,page_current);

			if(status_reg[4:2] == 3'b111)
			begin
				safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
				safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
				$display("%t: WARNING : Whole Portion (4MB) is Write Protected (Protected Addresses 000000h --> 3FFFFFh)\n", $realtime);
				$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
			end
			else if(status_reg[4:2] == 3'b110)
			begin
				if((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE/2))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower half of memory is Write Protected (Protected Addresses (000000h --> 1FFFFFh) =>(0 --> 2097151))\n",$realtime);
					$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
				end
				else
				begin
					if((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE/2))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper half of memory is Write Protected (Protected Addresses (200000h --> 3FFFFFh) =>(2097152 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
					end
				end
			end
			else if(status_reg[4:2] == 3'b101)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE*1/4))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower quarter of memory is Write Protected (Protected Addresses (000000h --> 0FFFFFh) =>(0 --> 1048575))",$realtime);
					$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n",$realtime, address_no);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE*3/4))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper quarter of memory is Write Protected (Protected Addresses (300000h --> 3FFFFFh) =>(3145728 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n",$realtime, address_no);
					end
				end
			end
			else if(status_reg[4:2] == 3'b100)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE*1/8))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower eighth of memory is Write Protected (Protected Addresses (000000h --> 07FFFFh) =>(0 --> 524287))\n",$realtime);
					$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE*7/8))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper eighth of memory is Write Protected (Protected Addresses (380000h --> 3FFFFFh) =>(3670016 --> 4194304))\n",$realtime);
						$display("%t:  ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
					end
				end
			end
			else if(status_reg[4:2] == 3'b011)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE*1/16))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower sixteenth of memory is Write Protected (Protected Addresses (000000h --> 03FFFFh) =>(0 --> 262143))\n",$realtime);
					$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE*15/16))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper sixteenth of memory is Write Protected (Protected Addresses (3C0000h --> 3FFFFFh) =>(3932160 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
					end
				end
			end
			else if(status_reg[4:2] == 3'b010)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE*1/32))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower thirty-second of memory is Write Protected (Protected Addresses (000000h --> 01FFFFh) =>(0 --> 131071))\n",$realtime);
					$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE*31/32))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING: Upper thirty-second of memory is Write Protected (Protected Addresses (3E0000h --> 3FFFFFh) =>(4063232 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE: The inserted address is %h that is inside the Protected memory portion!\n", $realtime, address_no);
					end
				end
			end

			else if(status_reg[4:2] == 3'b001)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE*1/64))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE SECTOR ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower sixty-fourth of memory is Write Protected (Protected Addresses (000000h --> 00FFFFh) =>(0 --> 65535))\n",$realtime);
					$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE*63/64))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; ////
						$display("%t: WARNING : Upper sixty-fourth of memory is Write Protected (Protected Addresses (3F0000h --> 3FFFFFh) =>(4128768 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The SECTOR ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
					end
				end
			end
			else begin
				status_reg[0] = 1'b1;
				#(M95XXX_SIM.M95XXX_Macro_mux.tSE);

				for(k=0; k <= (address_no); k=k+(8*`PAGE_SIZE))
				begin
					if(address_no >= k && address_no < (k+(8*`PAGE_SIZE)))
					begin
						no_address_start_sect = k;
						no_address_end_sect = k+(8*`PAGE_SIZE);
						page_start_sect = (no_address_start_sect /`PAGE_SIZE);
						page_end_sect = ((no_address_end_sect-1) /`PAGE_SIZE);
						sector_no = page_end_sect/8;
						$display("%t: NOTE : SECTOR[%d] has been involved in SECTOR ERASE instruction\n",$realtime,sector_no);
						$display("%t: NOTE : The sector started at address [%h] corresponding at PAGE[%d]\n",$realtime,no_address_start_sect,page_start_sect);
						$display("%t: NOTE : The sector finished at address[%h] corresponding at PAGE[%d]\n",$realtime,no_address_end_sect-1,page_end_sect);
					end

					while(no_address_start_sect < no_address_end_sect)
					begin
						memory[no_address_start_sect] = 'hFF;
						//$display("%t: Memory content is %h at address %h",$realtime,memory[no_address_start_sect],no_address_start_sect);
						no_address_start_sect = no_address_start_sect + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};
					end

					no_address_end_sect = no_address_start_sect-1;

					sector_inprogress = 1'b0;
					status_reg[0] = 1'b0;           //WIP is 0 when this cycle completed
					status_reg[1] = 1'b0;          //WEL must be set to 0 after the ERASE cycle
					safe_reg[5] = 1'b0;            //ERF=0 if the SECTOR ERASE instruction is verified successfully; otherwise ERF=1.



				end
				$display("%t: NOTE : The SECTOR ERASE instruction is finished successfully!!.",$realtime);
			//$display("%t: NOTE : SECTOR ERASE INSTRUCTION\n",$realtime);
			end
			k=0;
		end
//----------------------------------------------
//------------------ FOR BLOCK ERASE INSTRUCTION --------------!!!!!!!!!!!!!//
		if(operation == BKER)
		begin
			t_modify = $realtime;
			modify_operation = 1'b1;
			modify_operation_cher =1'b0;
			block_inprogress = 1'b1;

			data_byte_block = 1'b0;
			operation = 5'b00000;

			page_current_block = page_address_block;//page_address_sect;////MODIFICARE page_address_block
			address_no_block = start_address_block;//start_address_sect; ////MODIFICARE start_address_block
			$display("%t: WARNING : BLOCK ERASE INSTRUCTION is starting!\n",$realtime);
			$display("%t: NOTE : Inserted Address[%h] is located in PAGE[%d]\n",$realtime,address_no_block,page_current_block);

			if(status_reg[4:2] == 3'b111)
			begin
				safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
				safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
				$display("%t: WARNING : Whole Portion (4MB) is Write Protected (Protected Addresses 000000h --> 3FFFFFh)\n", $realtime);
				$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
			end
			else if(status_reg[4:2] == 3'b110)
			begin
				if((status_reg[6] == 1'b1)&&(address_no_block <= `MEM_SIZE/2))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower half of memory is Write Protected (Protected Addresses (000000h --> 1FFFFFh) =>(0 --> 2097151))\n",$realtime);
					$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no_block);
				end
				else
				begin
					if((status_reg[6] == 1'b0)&&(address_no_block >= `MEM_SIZE/2))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper half of memory is Write Protected (Protected Addresses (200000h --> 3FFFFFh) =>(2097152 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no_block);
					end
				end
			end
			else if(status_reg[4:2] == 3'b101)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no_block <= `MEM_SIZE*1/4))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower quarter of memory is Write Protected (Protected Addresses (000000h --> 0FFFFFh) =>(0 --> 1048575))",$realtime);
					$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n",$realtime, address_no_block);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no_block >= `MEM_SIZE*3/4))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper quarter of memory is Write Protected (Protected Addresses (300000h --> 3FFFFFh) =>(3145728 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n",$realtime, address_no_block);
					end
				end
			end
			else if(status_reg[4:2] == 3'b100)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no_block <= `MEM_SIZE*1/8))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower eighth of memory is Write Protected (Protected Addresses (000000h --> 07FFFFh) =>(0 --> 524287))\n",$realtime);
					$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no_block);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no_block >= `MEM_SIZE*7/8))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper eighth of memory is Write Protected (Protected Addresses (380000h --> 3FFFFFh) =>(3670016 --> 4194304))\n",$realtime);
						$display("%t:  ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no_block);
					end
				end
			end
			else if(status_reg[4:2] == 3'b011)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no_block <= `MEM_SIZE*1/16))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower sixteenth of memory is Write Protected (Protected Addresses (000000h --> 03FFFFh) =>(0 --> 262143))\n",$realtime);
					$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no_block);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no_block >= `MEM_SIZE*15/16))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper sixteenth of memory is Write Protected (Protected Addresses (3C0000h --> 3FFFFFh) =>(3932160 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no_block);
					end
				end
			end
			else if(status_reg[4:2] == 3'b010)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no_block <= `MEM_SIZE*1/32))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower thirty-second of memory is Write Protected (Protected Addresses (000000h --> 01FFFFh) =>(0 --> 131071))\n",$realtime);
					$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no_block);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no_block >= `MEM_SIZE*31/32))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING: Upper thirty-second of memory is Write Protected (Protected Addresses (3E0000h --> 3FFFFFh) =>(4063232 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE: The inserted address is %h that is inside the Protected memory portion!\n", $realtime, address_no_block);
					end
				end
			end

			else if(status_reg[4:2] == 3'b001)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no_block <= `MEM_SIZE*1/64))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower sixty-fourth of memory is Write Protected (Protected Addresses (000000h --> 00FFFFh) =>(0 --> 65535))\n",$realtime);
					$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no_block);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no_block >= `MEM_SIZE*63/64))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE BLOCK ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper sixty-fourth of memory is Write Protected (Protected Addresses (3F0000h --> 3FFFFFh) =>(4128768 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The BLOCK ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no_block);
					end
				end
			end
			else begin
				status_reg[0] = 1'b1;
				#(M95XXX_SIM.M95XXX_Macro_mux.tBE);

				for(k=0; k <= (address_no_block); k=k+(128*`PAGE_SIZE))
				begin
					if(address_no_block >= k && address_no_block < (k+(128*`PAGE_SIZE)))
					begin
						no_address_start_block = k;
						no_address_end_block = k+(128*`PAGE_SIZE);
						page_start_block = (no_address_start_block /(`PAGE_SIZE));
						page_end_block = ((no_address_end_block-1) /(`PAGE_SIZE));
						sector_start = page_start_block/8;
						sector_end = ((page_end_block+1)/8)-1;
						block_no = page_end_block/128;
						$display("%t: NOTE : BLOCK[%d] has been involved in BLOCK ERASE instruction\n",$realtime,block_no);
						$display("%t: NOTE : The block started at address [%h] corresponding at PAGE[%d]\n",$realtime,no_address_start_block,page_start_block);
						$display("%t: NOTE : The block finished at address[%h] corresponding at PAGE[%d]\n",$realtime,no_address_end_block-1,page_end_block);
						$display("%t: NOTE : SECTORS from [%d] to [%d] are involved!!\n",$realtime,sector_start,sector_end);
					end

					while(no_address_start_block < no_address_end_block)
					begin
						memory[no_address_start_block] = 'hFF;
						//$display("%t: Memory content is %h at address %h",$realtime,memory[no_address_start_block],no_address_start_block);
						no_address_start_block = no_address_start_block + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};
					end

					no_address_end_block = no_address_start_block-1;

					block_inprogress = 1'b0;
					status_reg[0] = 1'b0;           //WIP is 0 when this cycle completed
					status_reg[1] = 1'b0;          //WEL must be set to 0 after the ERASE cycle
					safe_reg[5] = 1'b0;            //ERF=0 if the SECTOR ERASE instruction is verified successfully; otherwise ERF=1.



				end
				$display("%t: NOTE : The BLOCK ERASE instruction is finished successfully!!.",$realtime);
			//$display("%t: NOTE : BLOCK ERASE INSTRUCTION\n",$realtime);
			end
			k=0;
		end

		//-------------  PAGE ERASE OPERATION
		if(operation == PGER)
		begin
			t_modify = $realtime;
			modify_operation = 1'b1;
			modify_operation_cher =1'b0;
			page_inprogress = 1'b1;
			data_byte_page = 1'b0;
			operation = 5'b00000;
			page_current = page_address_page;
			address_no = start_address_page;
			$display("%t: WARNING : PAGE ERASE INSTRUCTION is starting!\n",$realtime);
			$display("%t: NOTE : Inserted Address[%h] is located in PAGE[%d]\n",$realtime,address_no,page_current);

			if(status_reg[4:2] == 3'b111)
			begin
				safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
				safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
				$display("%t: WARNING : Whole Portion (4MB) is Write Protected (Protected Addresses 000000h --> 3FFFFFh)\n", $realtime);
				$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
			end
			else if(status_reg[4:2] == 3'b110)
			begin
				if((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE/2))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower half of memory is Write Protected (Protected Addresses (000000h --> 1FFFFFh) =>(0 --> 2097151))\n",$realtime);
					$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
				end
				else
				begin
					if((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE/2))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper half of memory is Write Protected (Protected Addresses (200000h --> 3FFFFFh) =>(2097152 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
					end
				end
			end
			else if(status_reg[4:2] == 3'b101)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE*1/4))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower quarter of memory is Write Protected (Protected Addresses (000000h --> 0FFFFFh) =>(0 --> 1048575))",$realtime);
					$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n",$realtime, address_no);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE*3/4))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper quarter of memory is Write Protected (Protected Addresses (300000h --> 3FFFFFh) =>(3145728 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n",$realtime, address_no);
					end
				end
			end
			else if(status_reg[4:2] == 3'b100)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE*1/8))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower eighth of memory is Write Protected (Protected Addresses (000000h --> 07FFFFh) =>(0 --> 524287))\n",$realtime);
					$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE*7/8))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper eighth of memory is Write Protected (Protected Addresses (380000h --> 3FFFFFh) =>(3670016 --> 4194304))\n",$realtime);
						$display("%t:  ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
					end
				end
			end
			else if(status_reg[4:2] == 3'b011)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE*1/16))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower sixteenth of memory is Write Protected (Protected Addresses (000000h --> 03FFFFh) =>(0 --> 262143))\n",$realtime);
					$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE*15/16))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper sixteenth of memory is Write Protected (Protected Addresses (3C0000h --> 3FFFFFh) =>(3932160 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
					end
				end
			end
			else if(status_reg[4:2] == 3'b010)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE*1/32))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower thirty-second of memory is Write Protected (Protected Addresses (000000h --> 01FFFFh) =>(0 --> 131071))\n",$realtime);
					$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE*31/32))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING: Upper thirty-second of memory is Write Protected (Protected Addresses (3E0000h --> 3FFFFFh) =>(4063232 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE: The inserted address is %h that is inside the Protected memory portion!\n", $realtime, address_no);
					end
				end
			end

			else if(status_reg[4:2] == 3'b001)
			begin
				if ((status_reg[6] == 1'b1)&&(address_no <= `MEM_SIZE*1/64))
				begin
					safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
					safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
					$display("%t: WARNING : Lower sixty-fourth of memory is Write Protected (Protected Addresses (000000h --> 00FFFFh) =>(0 --> 65535))\n",$realtime);
					$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
					$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
				end
				else
				begin
					if ((status_reg[6] == 1'b0)&&(address_no >= `MEM_SIZE*63/64))
					begin
						safe_reg[7] = 1'b1; //// A TEMPT TO MODIFY A PROTECTED MEMORY AREA HAS BEEN INDUCED
						safe_reg[5] = 1'b1; //// THE PAGE ERASE INSTRUCTION IS FAILED
						$display("%t: WARNING : Upper sixty-fourth of memory is Write Protected (Protected Addresses (3F0000h --> 3FFFFFh) =>(4128768 --> 4194304))\n",$realtime);
						$display("%t: ERROR : The PAGE ERASE instruction is not accepted!\n",$realtime);
						$display("%t: NOTE : The inserted address is %h that is inside the Protected memory portion!\n", $realtime,address_no);
					end
				end
			end
			else begin
				status_reg[0] = 1'b1;
				#(M95XXX_SIM.M95XXX_Macro_mux.tPE);

				for(k=0; k <= (address_no); k=k+`PAGE_SIZE)
				begin
					if(address_no >= k && address_no < (k+`PAGE_SIZE))
					begin
						no_address_start_page = k;
						no_address_end_page = k+`PAGE_SIZE;
					end

					while(no_address_start_page < no_address_end_page)
					begin
						memory[no_address_start_page] = 'hFF;
						//$display("%t: Memory content is %h at address %h",$realtime,memory[no_address_start_page],no_address_start_page);
						no_address_start_page = no_address_start_page + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};
					end

					no_address_end_page = no_address_start_page-1;

					page_inprogress = 1'b0;
					status_reg[0] = 1'b0;           //WIP is 0 when this cycle completed
					status_reg[1] = 1'b0;          //WEL must be set to 0 after the ERASE cycle
					safe_reg[5] = 1'b0;            //ERF=0 if the PAGE ERASE instruction is verified successfully; otherwise ERF=1.



				end
				$display("%t: NOTE : The PAGE ERASE instruction is finished successfully!!.",$realtime);
				$display("%t: /n=======================================================================",$realtime);
			//$display("%t: NOTE : PAGE ERASE INSTRUCTION\n",$realtime);
			end
			k=0;
		end
//----------------------------------------------

//==========================================================================///


		//----------------------------Write Data Instruction Execute
		if((operation == WRITE_DATA_IN)||(operation == WRITE_ID_DATA_IN)||(operation == PGPR) || (operation == PGPR_BUFF))
		begin
			t_modify = $realtime;
			//$display("%t MEMORY DEBUG: Writing data... Status_reg = %b",$realtime, status_reg);

			if (operation == PGPR_BUFF)
				begin
					$display("%t MEMORY: Writing data for PGPR_BUFF ",$realtime);
					if (status_reg[0] == 1'b0)
					begin
						$display("%t MEMORY: Buffer is already empty",$realtime);
						page_program_buff_data_end = 1'b1;
						no_wait = 1'b1;						
					end
				end
			else if (operation == PGPR)
				$display("%t MEMORY: Writing data for PGPR ",$realtime);
			else if (operation == WRITE_ID_DATA_IN)
				$display("%t MEMORY: Writing data for WRITE_ID_DATA_IN",$realtime);
			else if (operation == WRITE_DATA_IN)
				$display("%t MEMORY: Writing data for WRITE_DATA_IN ",$realtime);
			else 	
				$display("%t MEMORY else : Operation = %b ",$realtime, operation);

			$display("%t MEMORY: Status Register is %b \n \t \t \t \t \t \t  TB=%b | BP2=%b | BP1=%b | BP0=%b",$realtime,status_reg, status_reg[6],status_reg[4],status_reg[3],status_reg[2]);

			writemem_in_progress = 1'b1;
			modify_operation = 1'b1;
			modify_operation_cher =1'b0;
			data_byte = 1'b0;
			operation = 5'b00000;
			page_no = page_address;

			if(status_reg[4:2] == 3'b111) //regardeless of TB, the whole memory is write protected
			begin
				$display("%t: WARNING: Whole memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
				safe_reg[7] = 1'b1; //PAMAF = 1
				safe_reg[4] = 1'b1; //PRF
				if(write_data_end == 1'b1)
					safe_reg[5] = 1'b1; //ERF if PAGE WRITE INSTRUCTION
			end
			else if ((status_reg[6]) && (status_reg[4:2] != 3'b000)&& (page_no < `PAGES/2))//TB = 1  the memory is blocked from lower side
			begin
				//$display(" DEBUG page_no = %d     %d ", page_no, `PAGES/2);
				//safe_reg[7] = 1'b1; //PAMAF = 1
				//safe_reg[4] = 1'b1; //PRF
				if(write_data_end == 1'b1)
					safe_reg[5] = 1'b1; //ERF if PAGE WRITE INSTRUCTION
				if ((status_reg[4:2] == 3'b110)&&(page_no <= `PAGES/2)&&(!write_id_data_end)) //half memory is write protected, from 000000h to 1FFFFFh 2MB
				begin
					$display("%t: WARNING: Lower half of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 1FFFFFh are write protected, then address %h is not writable", memory_address);//page_no);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
				else if((status_reg[4:2] == 3'b101)&&(page_no <= `PAGES/4)&&(!write_id_data_end)) //a quarter of memory is write protected  1MB
				begin
					$display("%t: WARNING: Lower quarter of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 0FFFFFh are write protected, then address %h is not writable",  memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
				else if((status_reg[4:2] == 3'b100)&&(page_no <= `PAGES/8)&&(!write_id_data_end)) // 1/8 of memory is write protected  512KB
				begin
					$display("%t: WARNING: Lower eight of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 07FFFFhh are write protected, then address %h is not writable",  memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
				else if((status_reg[4:2] == 3'b011)&&(page_no <= `PAGES/16)&&(!write_id_data_end)) //1/16 of memory is write protected  256KB
				begin
					$display("%t: WARNING: Lower sixteenth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 03FFFFh are write protected, then address %h is not writable",  memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
				else if((status_reg[4:2] == 3'b010)&&(page_no <= `PAGES/32)&&(!write_id_data_end)) //1/32 of memory is write protected 128KB
				begin
					$display("%t: WARNING: Lower thirty-second of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 01FFFFh are write protected, then address %h is not writable",  memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
				else if((status_reg[4:2] == 3'b001)&&(page_no <= `PAGES/64)&&(!write_id_data_end)) //1/64 of memory is write protected  64KB
				begin
					$display("%t: WARNING: Lower sixty-fourth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 00FFFFh are write protected, then address %h is not writable",  memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
			end
			else if ((!status_reg[6]) && (status_reg[4:2] != 3'b000) &&(page_no >= `PAGES/2)) //TB =0  memory is blocked from upper side
			begin
//      safe_reg[7] = 1'b1; //PAMAF = 1
//      safe_reg[4] = 1'b1; //PRF
				if(write_data_end == 1'b1)
					safe_reg[5] = 1'b1; //ERF if PAGE WRITE INSTRUCTION
				if ((status_reg[4:2] == 3'b110)&&(page_no >= `PAGES/2)&&(!write_id_data_end)) //half memory is write protected, from 200000h to 3FFFFFh 2MB
				begin
					$display("%t: WARNING: upper half of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 200000h to 3FFFFFh are write protected, then address %h is not writable", memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
				else if((status_reg[4:2] == 3'b101)&&(page_no >= (`PAGES - (`PAGES/4)))&&(!write_id_data_end)) //a quarter of memory is write protected  1MB
				begin
					$display("%t: WARNING: Lower quarter of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
				else if((status_reg[4:2] == 3'b100)&&(page_no >= (`PAGES - (`PAGES/8)))&&(!write_id_data_end)) // 1/8 of memory is write protected  512KB
				begin
					$display("%t: WARNING: Lower eight of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
				else if((status_reg[4:2] == 3'b011)&&(page_no >= (`PAGES - (`PAGES/16)))&&(!write_id_data_end)) //1/16 of memory is write protected 256KB
				begin
					$display("%t: WARNING: Lower sixteenth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
				else if((status_reg[4:2] == 3'b010)&&(page_no >= (`PAGES - (`PAGES/32)))&&(!write_id_data_end)) //1/32 of memory is write protected 128KB
				begin
					$display("%t: WARNING: Lower thirty-second of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
				else if((status_reg[4:2] == 3'b001)&&(page_no >= (`PAGES - (`PAGES/64)))&&(!write_id_data_end)) //1/64 of memory is write protected  64KB
				begin
					$display("%t: WARNING: Lower sixty-fourth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
				end
			end
			else if((config_reg[0])&&(write_id_data_end))
			begin
				$display("%t: WARNING: ID PAGE is Permanently Locked, the Write Instruction ID Cycle has not been accepted!\n",$realtime);
				safe_reg[7] = 1'b1; //PAMAF = 1
			end

			else //if (status_reg[4:2] == 3'b000) //regardless of TB, the whole memory is writable, then start the write istruction
			begin
				//$display("%t DEBUG This Memory Area is not protected or you are in buffer mode",$realtime);
				if(write_data_byte_ok == 1'b1)
				begin  //Page Program
					write_data_byte_ok = 1'b0;
					if(write_id_data_end) $display("%t:MEMORY: NOTE: ID Page: Program Cycle has started! \n",$realtime);
					else if(page_program_data_end) $display("%t MEMORY: NOTE: Page[%d] Program Cycle has started! \n",$realtime, page_address);
					else if(page_program_buff_data_end) $display("%t MEMORY: NOTE: Page[%d] Program Cycle Buffer Mode has started! \n",$realtime, page_address);
					else  $display("%t:MEMORY: NOTE: Page[%d] Write Cycle has started! \n",$realtime,page_address);
					status_reg[0] = 1'b1;           //WIP is 1 during this cycle

					$display("%t MEMORY: status_reg now is %b",$realtime, status_reg);

					if (!page_program_data_end && !page_program_buff_data_end) //(!page_program_data_end) era funzionante! //PAGE WRITE INSTRUCTION
					begin : PAGE_WRITE_INSTRUCTION
								
						if(write_data_end)
							$display("%t MEMORY: Write Page Instruction started",$realtime);
						else if (write_id_data_end)
							$display("%t MEMORY: Write ID Page Instruction started",$realtime);

						#(M95XXX_SIM.M95XXX_Macro_mux.tPW);                         //Write Time
						if(start_address == end_address)  //Definite Page wrap around
						begin : loop1
							//$display( " DEBUG loop1");
							for(idx=1;idx<=`PAGE_SIZE;idx=idx+1)
							begin
								//$display("%t MEMORY DEBUG loop1: memory data = %h in the address %h ",$realtime, memory[{page_address,start_address}],{page_address,start_address});
								if(write_id_data_end)
								begin
									if(((write_id_data_end)&&(wrapping == 1'b1)) || (write_id_data_end)&&(memory_address[10]))
									begin
										wrapping = 1'b0;
										$display("%t MEMORY: Error: Cannot write beyond the boundary of the ID Page!",$realtime);
										$display("%t: Write ID PAGE instruction terminated",$realtime);
										disable loop1;
									end

									if(memory_address >= 24'h000 && memory_address < 24'h200)//!memory_address[9])  //memory_address > 3'h000 && memory_address <3'h200)//
									begin
										$display("%t MEMORY: Error: Cannot write from address [%h] because is in the first ID PAGE. You can write from address 200h to 3FFh. \n",$realtime,memory_address);
										$display("%t: Write ID PAGE instruction terminated",$realtime);
										disable loop1;
									end
									else if(!memory_address[10] && memory_address[9] && !wrapping)
									begin
										////$display(" DEBUG DATA LATCH = %b %h", data_latch[start_address],data_latch[start_address]);
										memory_id[{page_address[0],start_address}] = data_latch[start_address];
									end
								end
								else
								begin
									//$display(" DEBUG DATA LATCH = %b %h", data_latch[start_address],data_latch[start_address]);
									memory[{page_address,start_address}] = data_latch[start_address];

								end
								if(start_address == {`PAGE_OFFSET_BITS {1'b1}})
								begin
									start_address = {`PAGE_OFFSET_BITS {1'b0}};
									if(write_id_data_end) wrapping = 1'b1;
								end
								else start_address = start_address + {{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
							end //del for
						end //dell'if start_address==end_address
						else
						begin :loop2
							

							while(!(start_address == end_address))
							begin
								//$display(" DEBUG memory_address = [%h], write_id_data_end = %b , start_address = %h", memory_address,write_id_data_end,start_address);
								if(write_id_data_end)
								begin
									if((write_id_data_end)&&(wrapping == 1'b1) || (write_id_data_end)&&(memory_address[10]))
									begin
										wrapping = 1'b0;
										$display("%t: Error: Cannot write beyond the boundary of the ID Page!!",$realtime);
										$display("%t: Write ID PAGE instruction terminated \n======================================",$realtime);
										disable loop2;
									end

									if(memory_address >= 24'h000 && memory_address < 24'h200)//!memory_address[9])  //
									begin
										$display("%t MEMORY: Error: Cannot write from address [%h] because is in the first ID PAGE. You can write from address 200h to 3FFh. ",$realtime,memory_address);
										$display("%t: Write ID PAGE instruction terminated \n======================================",$realtime);
										disable loop2;
									end
									else if(!memory_address[10] && memory_address[9] && wrapping==1'b0)
									begin
										//$display(" DEBUG DATA LATCH = %b %h", data_latch[start_address],data_latch[start_address]);
										memory_id[{page_address[0],start_address}] = data_latch[start_address];
									end
								end
								else
								begin
									//$display(" DEBUG DATA LATCH = %b %h", data_latch[start_address],data_latch[start_address]);
									memory[{page_address,start_address}] = data_latch[start_address];
								end

								if(start_address == {`PAGE_OFFSET_BITS {1'b1}})
								begin
									start_address = {`PAGE_OFFSET_BITS {1'b0}};

									if(write_id_data_end) wrapping = 1'b1;
								end
								else
								begin
									start_address = start_address + 1; //{{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
								//$display(" DEBUG incremento indirizzi");
								end
							end //del while
						end //dell'begin loop2
					end //dell' if(operation ==WRITE DATA IN e ID DATA IN)

					else if (page_program_data_end)  //PAGE PROGRAM INSTRUCTION
					begin : PAGE_PROGRAM_INSTRUCTION
						$display("%t MEMORY: Page Program instruction is starting", $realtime);

						start_address_chk = {start_address[8:4],4'b0000}; //start check from the beginning of the 16 bits

						if(end_address[3:0] != 4'b0000)
						begin
							end_address_chk = end_address +(5'b10000 - end_address[3:0]); //end the check at the end of the 16 bits
//                                  $display("%t  DEBUG end_address_chk = %b |  end_address = %b ",$realtime, end_address_chk,end_address);
//                                  $display("%t MEMORY DEBUG: Check if addresses from [%h] to [%h] are in Erase Status",$realtime,{page_address,start_address_chk},{page_address,end_address_chk});
						end
						else
						begin
							end_address_chk = end_address;  //the end address is the last one of the 16 bits.
//                                  $display("%t DEBUG end_address_chk = %b == end_address = %b ",$realtime, end_address_chk,end_address);
//                                  $display("%t MEMORY DEBUG: Check if addresses from [%h] to [%h] are in Erase Status",$realtime,{page_address,start_address_chk},{(page_address+1'h1),end_address_chk});
						end

						//  $display("%t MEMORY DEBUG: Check if addresses from [%h] to [%h] are in Erase Status",$realtime,{page_address,start_address_chk},{page_address,end_address_chk});

						if (start_address_chk == end_address_chk)
						begin : loop_erase_chk
							//$display("%t DEBUG erased_status_chk %h ",$realtime,erased_status_chk);
							for(idx=1;idx<=`PAGE_SIZE;idx=idx+1)
							begin
								erased_status_chk = erased_status_chk & memory[{page_address,start_address_chk}];
								//$display("%t MEMORY DEBUG: erased_status_chk %h | data in memory %h", $realtime, erased_status_chk,memory[{page_address,start_address_chk}]);
								if(erased_status_chk != 8'hFF)
								begin
									//$display("%t MEMORY DEBUG: Page program operation failed, the selected memory addresses are already written. Do an erase instruction before. ",$realtime);
									$display("%t MEMORY: Page Program Failed because some bytes between %h and %h are not in erase state. Do an erase instruction before.",$realtime,{page_address,start_address_chk},{page_address,end_address_chk});
									safe_reg[4]= 1'b1; //PRF = 1;
									disable PAGE_PROGRAM_INSTRUCTION;
									disable loop_erase_chk;
								end
								if(start_address_chk == {`PAGE_OFFSET_BITS {1'b1}})
								begin
									start_address_chk = {`PAGE_OFFSET_BITS {1'b0}};
								end
								else
									start_address_chk = start_address_chk + 1; //{{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};

							end
						end //: loop_erase_chk
						else
						begin : loop_erase_chk2
							//$display("%t DEBUG start_address_chk %h [%b]| end_address_chk %h [%b]",$realtime,start_address_chk,start_address_chk,end_address_chk,end_address_chk);
							//$display("%t DEBUG erased_status_chk %h ",$realtime, erased_status_chk);
							while(!(start_address_chk == end_address_chk))
							begin
								erased_status_chk = erased_status_chk & memory[{page_address,start_address_chk}];
								//$display("%t MEMORY DEBUG: start_address_chk %h [%b] | end_address %h [%b]", $realtime,start_address_chk,start_address_chk,end_address,end_address);
								//$display("%t MEMORY DEBUG: erased_status_chk %h | data in memory %h", $realtime, erased_status_chk,memory[{page_address,start_address_chk}]);
								if(erased_status_chk != 8'hFF)
								begin
									$display("%t MEMORY: Page Program Failed because some bytes between %h and %h are not in erase state. Do an erase instruction before.",$realtime,{page_address,start_address_chk},{page_address,end_address_chk});
									safe_reg[4]= 1'b1; //PRF = 1;
									disable PAGE_PROGRAM_INSTRUCTION;
									disable loop_erase_chk2;
								end
								if(start_address_chk == {`PAGE_OFFSET_BITS {1'b1}})
								begin
									start_address_chk = {`PAGE_OFFSET_BITS {1'b0}};
								end
								else
									start_address_chk = start_address_chk + 1; //{{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
							end //del while
						end //: loop_erase_chk2


						if(end_address[3:0] != 4'b0000)
						begin
							$display("%t MEMORY: Memory addresses from [%h] to [%h] are in erase status. Page Progam cycle is starting",$realtime,{page_address,start_address[8:4],4'b0000},{(page_address),end_address_chk});//{start_address[8:4],4'b0000},end_address_chk);
						end
						else
						begin
							$display("%t MEMORY: Memory addresses from [%h] to [%h] are in erase status. Page Progam cycle is starting",$realtime,{page_address,start_address[8:4],4'b0000},{(page_address),end_address_chk-1'h1});//{start_address[8:4],4'b0000},end_address_chk);
						end


						#(M95XXX_SIM.M95XXX_Macro_mux.tPP); //1.5 ms



						if (erased_status_chk == 8'hFF)
						begin
							//$display("%t MEMORY DEBUG: WRITING .....................................",$realtime);
							//$display("%t MEMORY DEBUG: start_address = %h | end_address = %h",$realtime,start_address,end_address);
							if(start_address == end_address)  //Definite Page wrap around
							begin : loop3

								for(idx=1;idx<=`PAGE_SIZE;idx=idx+1)
								begin
//                                                  $display("%t MEMORY DEBUG loop3  idx = %d",$realtime, idx);
//                                                  $display("%t MEMORY DEBUG loop3: memory data = %h in the address %h ",$realtime, memory[{page_address,start_address}],{page_address,start_address});
									//$display(" DATA LATCH = %b %h", data_latch[start_address],data_latch[start_address]);
									memory[{page_address,start_address}] = data_latch[start_address];
//                                                  $display("%t MEMORY DEBUG loop3 :now data in memory in the address [%h]  is %h : data_latch [%h]  ", $realtime,{page_address,start_address}, memory[{page_address,start_address}],data_latch[start_address]);


									if(start_address == {`PAGE_OFFSET_BITS {1'b1}})
									begin
										start_address = {`PAGE_OFFSET_BITS {1'b0}};

									end
									else start_address = start_address + {{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
								end //del for
							end //del begin : loop1
							else
							begin :loop4
								while(!(start_address == end_address))
								begin
//                                                  $display("%t MEMORY DEBUG loop4: start_address = %h | end_address = %h",$realtime,start_address,end_address);
//                                                  $display("%t MEMORY DEBUG loop4: memory data = %h in the address %h ",$realtime, memory[{page_address,start_address}],{page_address,start_address});
									//$display(" DATA LATCH = %b %h", data_latch[start_address],data_latch[start_address]);
									memory[{page_address,start_address}] = data_latch[start_address];
//                                                      $display("%t MEMORY DEBUG loop4 :now data in memory in the address [%h]  is %h : data_latch [%h]  ", $realtime,{page_address,start_address}, memory[{page_address,start_address}],data_latch[start_address]);

									if(start_address == {`PAGE_OFFSET_BITS {1'b1}})
									begin
										start_address = {`PAGE_OFFSET_BITS {1'b0}};
									end
									else
										start_address = start_address + 1; //{{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
								end //del while
							end //dell'begin : loop2   */
						end
						else
						begin
							$display("%t MEMORY: Page Program Failed because some bytes between %h and %h are not in erase state",$realtime,start_address,end_address);
						end
					end//PAGE PROGRAM INSTRUCTION

					else if (page_program_buff_data_end)//page_program_buff_mode_end)  //PAGE PROGRAM_buffer INSTRUCTION
					begin : PAGE_PROGRAM_buffer_mode_INSTRUCTION2
						$display("%t MEMORY: Page Program Buffer Mode instruction is starting",$realtime);
						if(volatile_reg[1])
						begin : Writing_buffer_data_to_memory2

							page_no_buff = page_address_buff;

							if(status_reg[4:2] == 3'b111) //regardeless of TB, the whole memory is write protected
							begin
								$display("%t: WARNING: Whole memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
								safe_reg[7] = 1'b1; //PAMAF = 1
								safe_reg[4] = 1'b1; //PRF
								volatile_reg[0] = 1'b0;
								mem_buffer_full = 1'b0;

							end
							else if ((status_reg[6]) && (status_reg[4:2] != 3'b000))//TB = 1  the memory is blocked from lower side
							begin

								//safe_reg[7] = 1'b1; //PAMAF = 1
								//safe_reg[4] = 1'b1; //PRF

								if ((status_reg[4:2] == 3'b110)&&(page_no_buff <= `PAGES/2)) //half memory is write protected, from 000000h to 1FFFFFh 2MB
								begin
									$display("%t: WARNING: Lower half of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 1FFFFFh are write protected, then address %h is not writable", memory_address_buff);//page_no);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
								else if((status_reg[4:2] == 3'b101)&&(page_no_buff <= `PAGES/4)&&(!write_id_data_end)) //a quarter of memory is write protected  1MB
								begin
									$display("%t: WARNING: Lower quarter of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 0FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
								else if((status_reg[4:2] == 3'b100)&&(page_no_buff <= `PAGES/8)&&(!write_id_data_end)) // 1/8 of memory is write protected  512KB
								begin
									$display("%t: WARNING: Lower eight of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 07FFFFhh are write protected, then address %h is not writable",  memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
								else if((status_reg[4:2] == 3'b011)&&(page_no_buff <= `PAGES/16)&&(!write_id_data_end)) //1/16 of memory is write protected 256KB
								begin
									$display("%t: WARNING: Lower sixteenth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 03FFFFh are write protected, then address %h is not writable",  memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
								else if((status_reg[4:2] == 3'b010)&&(page_no_buff <= `PAGES/32)&&(!write_id_data_end)) //1/32 of memory is write protected 128KB
								begin
									$display("%t: WARNING: Lower thirty-second of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 01FFFFh are write protected, then address %h is not writable",  memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
								else if((status_reg[4:2] == 3'b001)&&(page_no_buff <= `PAGES/64)&&(!write_id_data_end)) //1/64 of memory is write protected  64KB
								begin
									$display("%t: WARNING: Lower sixty-fourth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 00FFFFh are write protected, then address %h is not writable",  memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
							end
							else if ((!status_reg[6]) && (status_reg[4:2] != 3'b000)) //TB =0  memory is blocked from upper side
							begin
//                                                              safe_reg[7] = 1'b1; //PAMAF = 1
//                                                              safe_reg[4] = 1'b1; //PRF

								if ((status_reg[4:2] == 3'b110)&&(page_no_buff >= `PAGES/2)&&(!write_id_data_end)) //half memory is write protected, from 200000h to 3FFFFFh 2MB
								begin
									$display("%t: WARNING: upper half of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 200000h to 3FFFFFh are write protected, then address %h is not writable", memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;

								end
								else if((status_reg[4:2] == 3'b101)&&(page_no_buff >= (`PAGES - (`PAGES/4)))&&(!write_id_data_end)) //a quarter of memory is write protected  1MB
								begin
									$display("%t: WARNING: Lower quarter of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
								else if((status_reg[4:2] == 3'b100)&&(page_no_buff >= (`PAGES - (`PAGES/8)))&&(!write_id_data_end)) // 1/8 of memory is write protected  512KB
								begin
									$display("%t: WARNING: Lower eight of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
								else if((status_reg[4:2] == 3'b011)&&(page_no_buff >= (`PAGES - (`PAGES/16)))&&(!write_id_data_end)) //1/16 of memory is write protected    256KB
								begin
									$display("%t: WARNING: Lower sixteenth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
								else if((status_reg[4:2] == 3'b010)&&(page_no_buff >= (`PAGES - (`PAGES/32)))&&(!write_id_data_end)) //1/32 of memory is write protected 128KB
								begin
									$display("%t: WARNING: Lower thirty-second of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
								else if((status_reg[4:2] == 3'b001)&&(page_no_buff >= (`PAGES - (`PAGES/64)))&&(!write_id_data_end)) //1/64 of memory is write protected  64KB
								begin
									$display("%t: WARNING: Lower sixty-fourth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
									$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
									safe_reg[7] = 1'b1; //PAMAF = 1
									safe_reg[4] = 1'b1; //PRF
									volatile_reg[0] = 1'b0;
									mem_buffer_full = 1'b0;
								end
							end
							else //memory is not protected
							begin

								if(no_wait == 1'b0)
								begin
									$display("%t MEMORY: Waiting memory is free",$realtime);

									wait(status_reg[0] == 1'b0); //WIP 0

									$display("%t MEMORY: Just finished writing in memory. Memory is free | WIP = %b ",$realtime, status_reg[0]);
									//added//
									page_program_buff_data_end = 1'b1;
									///////
								end
								no_wait = 1'b0;
								status_reg[0] = 1'b1; //WIP 1


								start_address_chk = {start_address_buff[8:4],4'b0000}; //start check from the beginning of the 16 bits


								if(end_address_buff[3:0] != 4'b0000)
								begin
									end_address_chk = end_address_buff +(5'b10000 - end_address_buff[3:0]); //end the check at the end of the 16 bits
									$display("%t MEMORY: Check if addresses from [%h] to [%h] are in Erase Status",$realtime,{page_address_buff,start_address_chk},{page_address_buff,end_address_chk});
								end
								else
								begin
									end_address_chk = end_address_buff; //the end address is the last one of the 16 bits.
									$display("%t MEMORY: Check if addresses from [%h] to [%h] are in Erase Status",$realtime,{page_address_buff,start_address_chk},{(page_address_buff),(end_address_chk-1'h1)});
								end



								if (start_address_chk == end_address_chk)
								begin : loop_erase_chk_3
									//$display(" DEBUG loop_erase_chk_3 PGPR");
									//$display("%t DEBUG erased_status_chk %h ",$realtime,erased_status_chk);
									for(idx=1;idx<=`PAGE_SIZE;idx=idx+1)
									begin
										erased_status_chk = erased_status_chk & memory[{page_address_buff,start_address_chk}];
										//$display("%t MEMORY DEBUG: erased_status_chk %h | data in memory %h | addr = %b [%h]", $realtime, erased_status_chk,memory[{page_address_buff,start_address_chk}],{page_address_buff,start_address_chk},{page_address_buff,start_address_chk});
										if(erased_status_chk != 8'hFF)
										begin
											//$display("%t MEMORY DEBUG: Page program operation failed, the selected memory addresses are already written. Do an erase instruction before. ",$realtime);
											$display("%t MEMORY: Page Program Failed because some bytes between %h and %h are not in erase state. Do an erase instruction before.",$realtime,{page_address_buff,start_address_chk},{page_address_buff,end_address_chk});
											safe_reg[4]= 1'b1; //PRF = 1;
											$display("%t: PRF = %b ",$realtime, safe_reg[4]);
											disable Writing_buffer_data_to_memory;
											disable loop_erase_chk_3;
										end
										if(start_address_chk == {`PAGE_OFFSET_BITS {1'b1}})
										begin
											start_address_chk = {`PAGE_OFFSET_BITS {1'b0}};
										end
										else
											start_address_chk = start_address_chk + 1; //{{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};

									end
								end //: loop_erase_chk
								else
								begin : loop_erase_chk4


									//$display("%t DEBUG start_address_chk %h [%b]| end_address_chk %h [%b]",$realtime,start_address_chk,start_address_chk,end_address_chk,end_address_chk);
									//$display("%t DEBUG erased_status_chk %h ",$realtime, erased_status_chk);
									while(!(start_address_chk == end_address_chk))
									begin
										erased_status_chk = erased_status_chk & memory[{page_address_buff,start_address_chk}];
										//$display("%t MEMORY DEBUG: start_address_chk %h [%b] | end_address %h [%b]", $realtime,start_address_chk,start_address_chk,end_address,end_address);
										//$display("%t MEMORY DEBUG: erased_status_chk %h | data in memory %h", $realtime, erased_status_chk,memory[{page_address,start_address_chk}]);
										//$display("%t MEMORY DEBUG: erased_status_chk %h | data in memory %h | addr = %b [%h]", $realtime, erased_status_chk,memory[{page_address_buff,start_address_chk}],{page_address_buff,start_address_chk},{page_address_buff,start_address_chk});

										if(erased_status_chk != 8'hFF)
										begin
											$display("%t MEMORY: Page Program Failed because some bytes between %h and %h are not in erase state. Do an erase instruction before.",$realtime,{page_address,start_address_chk},{page_address,end_address_chk});
											safe_reg[4]= 1'b1; //PRF = 1;
											$display("%t: PRF = %b ",$realtime, safe_reg[4]);
											disable Writing_buffer_data_to_memory;
											disable loop_erase_chk4;
										end
										if(start_address_chk == {`PAGE_OFFSET_BITS {1'b1}})
										begin
											start_address_chk = {`PAGE_OFFSET_BITS {1'b0}};
										end
										else
											start_address_chk = start_address_chk + 1; //{{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
									end //del while
								end //: loop_erase_chk2
								if(end_address[3:0] != 4'b0000)
								begin
									//$display(" DEBUG");
									$display("%t MEMORY: Memory addresses from [%h] to [%h] are in erase status. Writing the datas in the data latch",$realtime,{page_address_buff,start_address_buff[8:4],4'b0000},{(page_address_buff),end_address_chk-1'h1});//{start_address[8:4],4'b0000},end_address_chk);

								end
								else
								begin
									//$display(" DEBUG2");
									$display("%t MEMORY: Memory addresses from [%h] to [%h] are in erase status. Writing the datas in the data latch",$realtime,{page_address_buff,start_address_buff[8:4],4'b0000},{(page_address_buff),end_address_chk-1'h1});//{start_address[8:4],4'b0000},end_address_chk);
								end



								// ==================================================================================================================================

								$display("%t MEMORY: Loading Buffer Data   | WIP = %b", $realtime, status_reg[0]);


								memory_address = memory_address_buff;
								end_address = end_address_buff;
								start_address = start_address_buff;
								start_address_copy = start_address_buff;
								page_address = page_address_buff;
								num_bytes = (end_address - start_address)== 0 ? 512 : (end_address-start_address) ;
								//========
								$display("%t MEMORY: Writing %d data bytes from buffer in the datalatch",$realtime, num_bytes);
								incr = 0;

								while(incr < num_bytes)
								begin
									//$display("%t DEBUG sto scrivendo l'indirizzo %h  = %h ",$realtime,start_address_copy,{page_address_buff,start_address_copy});
									//$display("%t DEBUG sto scrivendo l'indirizzo %d  = %d ",$realtime,incr,num_bytes);
									//data_latch[start_address_copy] = mem_buffer[start_address_copy];
									data_latch[start_address_copy] = mem_buffer[start_address_copy];
									start_address_copy = start_address_copy + 1;
									incr = incr +1;
								//start_address_copy = start_address_copy + 1;
								end

								$display("%t MEMORY: Data from buffer have been written in the data latch, then memory buffer is empty",$realtime);

								volatile_reg[0] = 1'b0;
								mem_buffer_full = 1'b0;
								//$display("%t DEBUG vol_reg[0]2 = %b",$realtime,volatile_reg[0]);	
								//$display("%t DEBUG BUFLD = %b",$realtime, volatile_reg[0]);
								//status_reg[0] = 1'b1;
								//  ===================================================================================================================================

								$display("%t MEMORY: Writing in memory: 1.5ms from this moment! ",$realtime);
								#(M95XXX_SIM.M95XXX_Macro_mux.tPP); //1.5 ms

								//$display("%t DEBUG 2page_program_buff_data_end = %b ",$realtime, page_program_buff_data_end);

								//$display("%t DEBUG ovvero: \n memory_address = %b \n end_address = %b  \n start_address = %b \n page_address = %b ",$realtime,memory_address,end_address,start_address,page_address);


								if (erased_status_chk == 8'hFF)
								begin
									//$display("%t MEMORY DEBUG: WRITING .....................................",$realtime);
									//$display("%t MEMORY DEBUG: start_address = %h | end_address = %h",$realtime,start_address,end_address);
									if(start_address == end_address)  //Definite Page wrap around
									begin : loop5

										for(idx=1;idx<=`PAGE_SIZE;idx=idx+1)
										begin
											//$display("%t MEMORY DEBUG loop5  idx = %d",$realtime, idx);
											//$display("%t MEMORY DEBUG loop5: memory data = %h in the address %h ",$realtime, memory[{page_address,start_address}],{page_address,start_address});
											memory[{page_address,start_address}] = data_latch[start_address];
											//  $display("%t MEMORY DEBUG loop5 :now data in memory in the address [%h]  is %h : data_latch [%h]  ", $realtime,{page_address,start_address}, memory[{page_address,start_address}],data_latch[start_address]);

											if(start_address == {`PAGE_OFFSET_BITS {1'b1}})
											begin
												start_address = {`PAGE_OFFSET_BITS {1'b0}};

											end
											else start_address = start_address + {{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
										end //del for
									end //del begin : loop5
									else
									begin :loop6
										while(!(start_address == end_address))
										begin
											//$display("%t MEMORY DEBUG loop6: start_address = %h | end_address = %h",$realtime,start_address,end_address);
											//$display("%t MEMORY DEBUG loop6: memory data = %h in the address %h ",$realtime, memory[{page_address,start_address}],{page_address,start_address});
											memory[{page_address,start_address}] = data_latch[start_address];
											//$display("%t MEMORY DEBUG loop6 :now data in memory in the address [%h]  is %h : data_latch [%h]  ", $realtime,{page_address,start_address}, memory[{page_address,start_address}],data_latch[start_address]);

											if(start_address == {`PAGE_OFFSET_BITS {1'b1}})
											begin
												start_address = {`PAGE_OFFSET_BITS {1'b0}};
											end
											else
												start_address = start_address + 1; //{{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
										end //del while
									end //dell'begin : loop6   */
								end
								else
								begin
									$display("%t MEMORY: Page Program Failed because some bytes between %h and %h are not in erase state",$realtime,start_address,end_address);
								end
							end //else (memory not protected)
						end//Writing_buffer_data_to_memory


					end //PAGE PROGRAM_buffer INSTRUCTION

					while(rds_inprogress == 1 && i !== 7) //read status goes directly to 0
					begin : loop
						#(10);
					end

					//$display("%t MEMORY DEBUG: STATUS_REGISTER = %b",$realtime, status_reg);
					if(write_id_data_end&&!memory_address[10]) $display("%t:MEMORY:  NOTE: ID Page: Program Cycle is completed! \n",$realtime);
					else if (page_program_data_end)     $display("+++++++++++++++++++++++++++++++++++++\n%t MEMORY:  NOTE: Page Program Operation is finished!\n++++++++++++++++++++++++++++++++++\n\n",$realtime);
					else if (page_program_buff_data_end)
					begin
						$display("%t ++++++++++++++++++++++++++++++++++++++++++++++++++++++++",$realtime);
						$display("%t MEMORY:  NOTE: Page Program Buffer Mode Operation is finished! ",$realtime);
						$display("%t ++++++++++++++++++++++++++++++++++++++++++++++++++++++++ \n \n ",$realtime);
					end
					else if (write_data_end) $display("%t MEMORY: NOTE: Page[%d] Write Cycle is completed! \n",$realtime,page_address);
					else    $display("+++++++++++++++++++\n%t ERROR\n+++++++++++++++++++",$realtime,$realtime);



					if(!volatile_reg[1])//if Buffer Mode is disabled
					begin
						status_reg[1] = 1'b0;           //WEL is reset
					end

					status_reg[0] = 1'b0;           //WIP is 0 when this cycle completed


					//$display("%t MEMORY DEBUG: STATUS_REGISTER =  %b \n",$realtime, status_reg);
					wrapping = 1'b0;
				end
				else if(write_data_byte_ok == 1'b0)
					$display("%t: ERROR: \"/S\" WRITE should not be deselected after the 8th bit of data is latched, Page Program instruction is not executed.\n",$realtime);
			end

		end
		write_id_data_end = 1'b0;
		write_data_end = 1'b0;
		page_program_data_end = 1'b0;
		page_program_buff_data_end = 1'b0;
		wr_protect = 1'b0;
		erased_status_chk = 8'hFF;


		if(S && (!deep_pwr_active))
		begin

			mode = device_standby_mode; dout = 1'bz;
		end

		writemem_in_progress = 1'b0;

	end



	always@(posedge byte_ok)
	begin
		//----------------------------instruction byte input and decode, needed for clk left high at end of transaction
		if((mode == active_power_mode)&&(instruction_byte == 1'b1)&&(byte_ok == 1'b1))
		begin
			instruction_code = shift_in_reg;

			casex(instruction_code)
				8'b0000_x110:   begin
					instruction = WRITE_ENABLE;
					instruction_byte = 1'b0; byte_ok = 1'b0;

					instruction = 5'b0000;
					if(status_reg[0] == 1'b1)   //WIP is set, an internal write cycle is in progress
						$display("%t: WARNING:  An internal write cycle is in progress, this WRITE ENABLE INSTRUCTION is rejected!",$realtime);
					else begin
						write_en_id = 1'b1;
						operation = EN_WRITE;
						$display("%t: WARNING: WRITE ENABLE INSTRUCTION is starting!",$realtime);
					end

				end
				8'b0000_x100:   begin
					instruction = WRITE_DISABLE;
					instruction_byte = 1'b0; byte_ok = 1'b0;
					instruction_code = shift_in_reg;

					instruction = 5'b0000;
					if(status_reg[0] == 1'b1)   //WIP is set, an internal write cycle is in progress
						$display("%t: WARNING:  An internal write cycle is in progress, this WRITE DISABLE INSTRUCTION is rejected!",$realtime);
					else begin
						write_dis_id = 1'b1;
						operation = DIS_WRITE;
					end
				end
			endcase
		end
	end



//=========================================================
	always@(negedge clk_in)
	begin

//$display("MEMORY DEBUG: negedge CLK_IN");

//=======================
		//----------------------------instruction byte input and decode
		if((mode == active_power_mode)&&(instruction_byte == 1'b1)&&(byte_ok == 1'b1))
		begin
			instruction_byte = 1'b0; byte_ok = 1'b0;
			instruction_code = shift_in_reg;
			$display("\n%t MEMORY: Instruction Code acquired: %h [%b]",$realtime,instruction_code,instruction_code);


			if((!volatile_reg[1]) && (!deep_pwr_active))  //if buffer mode is inactive, all Instructions can be decoded
			begin
				case(instruction_code)
					8'b0000_0110:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the WRITE ENABLE instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = WRITE_ENABLE; end
					8'b0000_0100:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the WRITE DISABLE instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = WRITE_DISABLE; end
					8'b0000_0001:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the WRITE Status Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = WRITE_STATUS_REGISTER; end
					8'b0000_0011:  begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the READ from memory array instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = READ_FROM_MEMORY_ARRAY; end
					8'b0000_0010:  begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the WRITE to memory array instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = WRITE_TO_MEMORY_ARRAY; end //page write (erase and program)
					8'b0000_0101:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the READ Status Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = READ_STATUS_REGISTER; rds_inprogress = 1'b1; i = 8; end
					8'b1000_0011:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the READ Identification Page instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = READ_ID_PAGE; end
					8'b1000_0010:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the WRITE Identification Page instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = WRITE_ID_PAGE; end
					// new features
					8'b0000_1011:  begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the FAST READ Data instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = FAST_READ_DATA ; end//fast read single output with 1 dummy byte
					8'b0011_1011:  begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the FAST DUAL READ Data instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = FAST_DUAL_READ_DATA; end //fast read dual output with 1 dummy byte
					8'b0110_1011:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the FAST QUAD READ Data instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = FAST_QUAD_READ_DATA;end //fast read quad output with 1 dummy byte
					8'b0000_1010:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the PAGE PROGRAM instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = PAGE_PROGRAM ; //page program
						start_PGPR = 1'b1;
					end
					8'b1101_1011:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the PAGE ERASE instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = PAGE_ERASE ; end//page erase
					8'b0010_0000: begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the SECTOR ERASE instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = SECTOR_ERASE ; end//sector erase
					8'b1101_1000: begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the BLOCK ERASE instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = BLOCK_ERASE ; end //block erase
					8'b1100_0111:  begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the CHIP ERASE instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = CHIP_ERASE ;end //chip erase
					8'b1000_1011:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the FAST READ Identification Page instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = FAST_READ_ID ; end//fast read identification id
					8'b1011_1001:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the DEEP POWER ENABLE instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = DEEP_POWER_ENTER ;end //deep power - down enter
					8'b1010_1011:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the DEEP POWER instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = RELEASE_DEEP_POWER ; end//deep power down release (wake up)
					8'b1001_1111:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the JEDEC instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = JEDID; jedid_inprogress = 1'b1; i =24;
						$display("%t MEMORY : JEDEC OPERATION START", $realtime);
					end//JEDEC Identification (SF)
					8'b0001_0101:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the READ CONFIGURATION SAFETY Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = READ_CONFIG_SAFETY_REG ; //read configuration and safety register
						read_config_safe_inprogress = 1'b1; i = 16;
						$display("%t MEMORY: READ CONFIGURATION & SAFETY REGISTER OPERATION START",$realtime);
					end
					8'b1000_0101:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the READ Volatile Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = READ_VOLATILE_REG;
						rdv_inprogress = 1'b1; i = 8;
					end //read volatile register
					8'b1000_0001:  begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the WRITE Volatile Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = WRITE_VOLATILE_REG ;end //write volatile register
					8'b0101_0000:  begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the CLEAR Safety Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = CLEAR_SAFETY_STI_FLAGS; end//clear safety sticky flags
					8'b0101_1010: begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the READ SFDP Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = READ_SFDP ; end//read SFDP

					8'b0110_0110:   instruction = RESET_ENABLE ; //enable reset
					8'b1001_1001:   instruction = RESET_SOFTWARE ; //software reset
					default:
						$display("%t: ERROR: The input instruction code[%b] is undefined!\n",$realtime,instruction_code);
				endcase
			end
			else if ((volatile_reg[1]) && (!deep_pwr_active)) //if buffer mode is active, only these instructions can be decoded
			begin
				case(instruction_code)
					8'b0000_0110:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the WRITE ENABLE instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = WRITE_ENABLE;end
					8'b0000_0100:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the WRITE DISABLE instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = WRITE_DISABLE; end
					8'b1000_0001:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the WRITE Volatile Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = WRITE_VOLATILE_REG ;end //write volatile register
					8'b1000_0101:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the READ Volatile Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = READ_VOLATILE_REG;
						rdv_inprogress = 1'b1; i = 8;
					end //read volatile register
					8'b0000_1010:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the PAGE PROGRAM instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = PAGE_PROGRAM ; //page program
						start_PGPR = 1'b1;
					end
					8'b0000_0101:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the READ Status Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = READ_STATUS_REGISTER; rds_inprogress = 1'b1; i = 8; end
					8'b0001_0101:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the READ CONFIGURATION SAFETY Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = READ_CONFIG_SAFETY_REG ; //read configuration and safety register
						read_config_safe_inprogress = 1'b1; i = 16;
						$display("%t MEMORY: READ CONFIGURATION & SAFETY REGISTER OPERATION START",$realtime);
					end
					8'b0101_0000:   begin
						if(reset_en_bit)
						begin
							$display("%t WARNING: The RESET ENABLE instruction is aborted because the next instruction is the CLEAR Safety Register instruction and not the SOFTWARE RESET instruction!\n", $realtime);
							reset_en_bit =1'b0;
						end
						instruction = CLEAR_SAFETY_STI_FLAGS; end //clear safety sticky flags
					8'b0110_0110:   instruction = RESET_ENABLE ; //enable reset
					8'b1001_1001:   instruction = RESET_SOFTWARE ; //software reset

					default:
						$display("%t: ERROR: Buffer Mode is active.\nThe input instruction code[%b] cannot be decoded!\n",$realtime,instruction_code);
				endcase
			end
			else if (deep_pwr_active)
			begin
				case (instruction_code)
					8'b1010_1011: begin instruction = RELEASE_DEEP_POWER;
					end

					8'b0110_0110:   instruction = RESET_ENABLE ; //enable reset
					8'b1001_1001:   instruction = RESET_SOFTWARE ; //software reset

					default:
					begin
						$display("%t: ERROR: Device is in deep power mode.\nThe input instruction code[%b] cannot be decoded!\n",$realtime,instruction_code);
						mode = deep_power_mode;
						//$display("%t: mode %b",$realtime,mode);
						disable attivazione_pgpr;
					end
				endcase
			end
		end
//=======================

		//----------------------------address high byte 2 input
		if((mode == active_power_mode)&&(address_h2_byte == 1'b1)&&(byte_ok == 1'b1))
		begin
			address_h2_byte = 1'b0; byte_ok = 1'b0;
			address_h2_code = shift_in_reg;
			address_h_byte =1'b1;
			bit_counter_en = 1'b1;
			bit_counter_ld = 1'b1;
		//$display("%t MEMORY: ADDRESS high byte 2 registering: %b",$realtime, shift_in_reg);
		end

//=======================
		//----------------------------address high byte 1 input
		if((mode == active_power_mode)&&(address_h_byte == 1'b1)&&(byte_ok == 1'b1))
		begin
			address_h_byte = 1'b0; byte_ok = 1'b0;
			address_h_code = shift_in_reg;
			address_l_byte =1'b1;
			bit_counter_en = 1'b1;
			bit_counter_ld = 1'b1;
		//$display("%t MEMORY: ADDRESS high byte 1 registering: %b",$realtime, shift_in_reg);
		end

//=======================
		//----------------------------address low byte input
		if((mode == active_power_mode)&&(address_l_byte == 1'b1)&&(byte_ok == 1'b1))
		begin
			address_l_byte = 1'b0; byte_ok = 1'b0;
			address_l_code = shift_in_reg;
			//$display("%t MEMORY: ADDRESS low byte registering: %b",$realtime, shift_in_reg);
			if (`MEM_ADDR_BITS <= 8)
				memory_address = {address_l_code};
			else if (`MEM_ADDR_BITS == 9)
				memory_address = {instruction_code[3],address_l_code};
			else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
				memory_address = {address_h_code,address_l_code};
			else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
			begin

				if(!page_prog_buff_mode_start || ((!status_reg[0] && !volatile_reg[0])) || !volatile_reg[1])
				begin
					//$display("%t Memory DEBUG Indirizzo precedentel nel memory_address %b = %h", $realtime,memory_address,memory_address);
					memory_address = {address_h2_code,address_h_code,address_l_code};
					//$display("%t Memory DEBUG indirizzo salvato nel memory_address %b = %h", $realtime,memory_address,memory_address);
					$display("%t MEMORY: Address acquired is : [%h] ",$realtime,memory_address);
				end

				else if(page_prog_buff_mode_start == 1'b1)
				begin
					//$display("%t Memory DEBUG Indirizzo precedente nel memory_address_buff %b = %h ",$realtime,memory_address_buff,memory_address_buff);
					memory_address_buff = {address_h2_code,address_h_code,address_l_code};
					//$display("%t Memory DEBUG indirizzo salvato nel memory_address_buff %b = %h ",$realtime,memory_address_buff,memory_address_buff);
					$display("%t MEMORY: Address acquired in the buffer is : [%h] ",$realtime,memory_address_buff);
				end
				else
				begin
					// $display("%t Memory DEBUG Indirizzo precedentel nel memory_address %b = %h", $realtime,memory_address,memory_address);
					// memory_address = {address_h2_code,address_h_code,address_l_code};
					// $display("%t Memory DEBUG indirizzo salvato nel memory_address %b = %h", $realtime,memory_address,memory_address);
					// $display("%t MEMORY DEBUG: Address acquired is : [%h] ",$realtime,memory_address);
					$display("%t Issue saving the memory address",$realtime);
				end

			end

//=======================
			//-------------------------  READ MEM OR ID ARRAY
			if(read_data_bytes == 1'b1) //receives address in Read Data Bytes Instruction
			begin
				if(tDiff_clk < 20.0) begin
					$display("\n%t xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx",$realtime);
					$display("%t MEMORY ERROR! Frequency is %g MHz, the READ DATA SINGLE OUTPUT instruction work at max 50MHz!",$realtime,(1000.0/tDiff_clk));
					$display("%t xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx\n",$realtime);
					read_data_bytes = 1'b0;
					#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid
					dout = 1'bx;
				end	
				else begin	
					read_data_bytes = 1'b0; operation = READ_DATA_OUT; i = 8;
					$display("%t MEMORY : READING DATA FROM ADDRESS [%h] STARTED",$realtime,memory_address);
				end
			end

//=======================

			if(fast_read_data_bytes == 1'b1) //receives address in Fast Quad Read Data Bytes Instruction
			begin
				fast_read_data_bytes = 1'b0; operation = FREAD; i = 8;
				$display("%t MEMORY: FREAD operation starting",$realtime);


				$display("%t MEMORY: DUMMY BYTE IS STARTING (400ns)",$realtime);
				#((M95XXX_SIM.M95XXX_Macro_mux.tC)*8);// DUMMY BYTE TIME WAITING
				$display("%t MEMORY: DUMMY BYTE ENDED",$realtime);
			end

//=======================

			if(fd_read_data_bytes == 1'b1) //receives address in Fast Quad Read Data Bytes Instruction
			begin
				fd_read_data_bytes = 1'b0; operation = FDREAD; i = 4;
				$display("%t MEMORY: FDREAD operation starting",$realtime);
				Qout_en_dual = 1'b1;

				$display("%t MEMORY: DUMMY BYTE IS STARTING (400ns)",$realtime);
				#((M95XXX_SIM.M95XXX_Macro_mux.tC)*8);// DUMMY BYTE TIME WAITING
				$display("%t MEMORY: DUMMY BYTE ENDED",$realtime);
			end

//=======================

			if(fq_read_data_bytes == 1'b1) //receives address in Fast Quad Read Data Bytes Instruction
			begin
				fq_read_data_bytes = 1'b0; operation = FQREAD; i = 2;
				$display("%t MEMORY: FQREAD operation starting",$realtime);
				Qout_en = 1'b1;
				$display("%t MEMORY: DUMMY BYTE IS STARTING (400ns)",$realtime);
				#((M95XXX_SIM.M95XXX_Macro_mux.tC)*8);// DUMMY BYTE TIME WAITING
				$display("%t MEMORY: DUMMY BYTE ENDED",$realtime);
			end

//=======================

			if(read_id_data_bytes == 1'b1)
			begin
				if(tDiff_clk < 20.0) begin
					$display("\n%t xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx",$realtime);
					$display("%t MEMORY ERROR! Frequency is %g MHz, the READ IDENTIFICATION instruction work at max 50MHz!",$realtime,(1000.0/tDiff_clk));
					$display("%t xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx\n",$realtime);
						read_id_data_bytes = 1'b0;
					#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid
					dout = 1'bx;
				end	
				else begin	
					if(memory_address < 'h03FF) //address h03FF = 1023
					begin
						read_id_data_bytes = 1'b0; operation = READ_ID_DATA_OUT; i = 8;
						//$display("%t MEMORY DEBUG : memory address = %h %b ",$realtime, memory_address,memory_address);
					end
					else
					begin
						read_id_data_bytes = 1'b0;
						$display("%t xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx",$realtime);
						$display("%t MEMORY: ERROR:  Memory Address %h has exceeded the ID memory dimension [000 - 3ff]", $realtime, memory_address);
						$display("%t MEMORY: READ ID DATA instruction failed\n",$realtime);
						$display("%t xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx",$realtime);
						#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);
					end
				end
			end

//=======================
//=========== FOR FAST READ IDENTIFICATION PAGE ======//
			if(fast_read_id_data_bytes == 1'b1)
			begin
				if( memory_address < 'h03FF)
				begin
					fast_read_id_data_bytes = 1'b0; operation = FRDID; i = 8;
					$display("%t MEMORY: DUMMY BYTE IS STARTING (%g ns)",$realtime, ((M95XXX_SIM.M95XXX_Macro_mux.tC)*8));
					#((M95XXX_SIM.M95XXX_Macro_mux.tC)*8);// DUMMY BYTE TIME WAITING
					$display("%t MEMORY: DUMMY BYTE ENDED",$realtime);
				end
				else
				begin
					fast_read_id_data_bytes = 1'b0;
					$display("%t MEMORY: ERROR: Memory Address %h has exceeded the ID memory dimension [000 - 3ff]", $realtime, memory_address);
					$display("\n%t FAST READ ID DATA instruction failed\n",$realtime);
					#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);
				end
			end
//======================================================//
//============ FOR READ_SFDP REGISTER =================//
			if(sfdp_data_bytes == 1'b1)
			begin
				if(memory_address < (`PAGE_SIZE -1))
				begin
					sfdp_data_bytes = 1'b0; operation = RDSFDP; i = 8;
					$display("%t MEMORY: DUMMY BYTE IS STARTING (400ns)",$realtime);
					#((M95XXX_SIM.M95XXX_Macro_mux.tC)*8);// DUMMY BYTE TIME WAITING
					$display("%t MEMORY: DUMMY BYTE ENDED",$realtime);
				end
				else
				begin
					sfdp_data_bytes = 1'b0;
					$display("%t MEMORY: ERROR: Inserted Address %h has exceeded the SFDP Register dimension [000 - 1ff]", $realtime, memory_address);
					$display("\n%t READ SFDP_Register instruction failed\n",$realtime);
					#(M95XXX_SIM.M95XXX_Macro_mux.tSHQZ);
				end
			end
//======================================================//

			if(write_id_data_in == 1'b1)   //receives address in Write ID Data Bytes Instruction
			begin
				if(config_reg[0]) //LOCK ID BIT ON CONFIG REGISTER
				begin
					safe_reg[7] = 1;
					safe_reg[4] = 1;
					$display("%t WARNING:  The ID PAGE is locked, Write ID Instruction Failed.   LID bit = %b ",$realtime, config_reg[0]);

				end
				else
				begin
					bit_counter_en = 1'b1; bit_counter_ld = 1'b1;
					write_id_data_in = 1'b0;
					write_id_data_end = 1'b1;
					operation = WRITE_ID_DATA_IN;
					data_byte = 1'b1;
					start_address = memory_address[`PAGE_OFFSET_BITS-1:0];
					page_offset_addr = memory_address[`PAGE_OFFSET_BITS-1:0];
					page_address = memory_address[`MEM_ADDR_BITS-1:`MEM_ADDR_BITS-`PAGE_ADDR_BITS];
				// $display("%t MEMORY DEBUG: start_address = [%b] \n page_offset_addr = [%b] \n page_address = [%b]",$realtime,start_address,page_offset_addr,page_address);
				end
			end

//=======================

			if(write_data_in == 1'b1)    //receives address in Write Data Bytes Instruction
			begin
				bit_counter_en = 1'b1; bit_counter_ld = 1'b1;
				write_data_in = 1'b0;
				write_data_end = 1'b1;
				operation = WRITE_DATA_IN;
				page_address = memory_address[`MEM_ADDR_BITS-1:`MEM_ADDR_BITS-`PAGE_ADDR_BITS];
				data_byte = 1'b1;
				start_address = memory_address[`PAGE_OFFSET_BITS-1:0];
				page_offset_addr = memory_address[`PAGE_OFFSET_BITS-1:0];
			end



//=======================

/////
			if(page_program_data_in == 1'b1)    //receives address in page program Data Bytes Instruction
			begin
				//$display("%t MEMORY DEBUG 30 page_prog_buff_mode_start = %b",$realtime,page_prog_buff_mode_start);

				if(page_prog_buff_mode_start ) //NON e' LA PRIMA VOLTA
				begin
					//$display("\n ==================%t DEBUG SONO NEI PASSAGGI SUCCESSIVI e WIP = %b BUFLD = %b",$realtime, status_reg[0],volatile_reg[0]);
					//$display("\t==========================================");
					//LA MEMORIA E' OCCUPATA?
					if(!status_reg[0]) //NO - WIP 0
					begin
						//il buffer e' pieno?
						if(!volatile_reg[0]) //NO - BUFLD 0
						begin
							$display("\n%t MEMORY: Memory is free (WIP = %b), Buffer is empty (BUFLD = %b). Page Program is writing in memory",$realtime,status_reg[0],volatile_reg[0]);
							$display("\t==========================================");
							//$display("%t MEMORY 1840: alzo i flag ecc e avvio l'operation ",$realtime);
							bit_counter_en = 1'b1; bit_counter_ld = 1'b1;
							erased_status_chk = 8'hFF;
							page_program_data_in = 1'b0;
							page_program_data_end = 1'b1;

							operation = PGPR;
							page_address = memory_address[`MEM_ADDR_BITS-1:`MEM_ADDR_BITS-`PAGE_ADDR_BITS];
							data_byte = 1'b1;
							start_address = memory_address[`PAGE_OFFSET_BITS-1:0];
							page_offset_addr = memory_address[`PAGE_OFFSET_BITS-1:0];
						//$display("%t MEMORY DEBUG 1840: finito la parte di avvio operation",$realtime);
						end  //fatta (forse)

						else if (volatile_reg[0]) //SI - BUFLD 1
						begin
							$display("\t==========================================");
							$display("%t MEMORY: Memory is free (WIP = %b) but Buffer is full! (BUFLD = %b) Writing data from buffer to the memory.",$realtime,status_reg[0],volatile_reg[0]);
							$display("\t==========================================");
						//  volatile_reg[0] = 1'b0; //Pongo il BUFLD a 0 dato che sto svuotando il buffer.
						//  data_latch = mem_buffer;

						end
						else
							$display("%t BUFLD %b is undefined",$realtime,volatile_reg[0]);
					end
					else if(status_reg[0]) //YES - WIP 1
					begin
						//The buffer is full?
						if(!volatile_reg[0]) //NO - BUFLD 0
						begin
							$display("\t==========================================");
							$display("%t MEMORY Memory is busy (WIP = %b) and Buffer is empty (BUFLD = %b). Writing in the buffer!",$realtime,status_reg[0],volatile_reg[0]);
							$display("\t==========================================");
							bit_counter_en = 1'b1; bit_counter_ld = 1'b1;
							erased_status_chk = 8'hFF;
							page_program_data_in = 1'b0;
							page_program_buff_data_end = 1'b1;

							operation = PGPR_BUFF;

							page_address_buff = memory_address_buff[`MEM_ADDR_BITS-1:`MEM_ADDR_BITS-`PAGE_ADDR_BITS];
							data_byte = 1'b1;
							data_byte_buff = 1'b1;
							start_address_buff = memory_address_buff[`PAGE_OFFSET_BITS-1:0];
							page_offset_addr_buff = memory_address_buff[`PAGE_OFFSET_BITS-1:0];

//                              $display("%t MEMORY DEBUG 30 operation = %b",$realtime,operation);
						//$display("%t MEMORY DEBUG 30 finito di salvare il memory_address_buff",$realtime);
//                              $display("\t DEBUG==========================================");
						end
						else if(volatile_reg[0]) //YES - BUFLD 1
						begin
							$display("\t==========================================");
							//$display("%t MEMORY DEBUG: Memory is busy and buffer is full",$realtime);
							$display("%t MEMORY: ERROR Memory is busy and Buffer is full, wait untill it will be empty! ",$realtime);
							$display("\t==========================================");
						end
						else
							$display("%t BUFLD %b is undefined",$realtime,volatile_reg[0]);

					end
					else
						$display("%t ERROR: WIP is undefined",$realtime);

				end

				else if(!page_prog_buff_mode_start)// || !volatile_reg[1] || (!status_reg[0] && !volatile_reg[0]))
				//nel caso normale, o nel caso sia la prima volta della buffer mode, o nel caso in cui la memoria
				//e' libera (wip=0) e il buffer e' anche libero (BUFLD=0), scrivo direttamente in memoria.
				begin

					//$display("%t MEMORY DEBUG 1840: alzo i flag ecc e avvio l'operation ",$realtime);
					bit_counter_en = 1'b1; bit_counter_ld = 1'b1;
					erased_status_chk = 8'hFF;
					page_program_data_in = 1'b0;
					page_program_data_end = 1'b1;

					operation = PGPR;
					page_address = memory_address[`MEM_ADDR_BITS-1:`MEM_ADDR_BITS-`PAGE_ADDR_BITS];
					data_byte = 1'b1;
					start_address = memory_address[`PAGE_OFFSET_BITS-1:0];
					page_offset_addr = memory_address[`PAGE_OFFSET_BITS-1:0];
				// $display("%t MEMORY DEBUG 1840: finito la parte di avvio operation",$realtime);

				end
			end

//=======================


//========================================= BLOCK ERASE INSTRUCTION ==========//
			if(block_erase_en == 1'b1)    //receives address in Write Data Bytes Instruction
			begin
				bit_counter_en = 1'b1;
				bit_counter_ld = 1'b1;
				block_erase_en = 1'b0;
				operation = BKER;
				page_address_block = memory_address[`MEM_ADDR_BITS-1:`MEM_ADDR_BITS-`PAGE_ADDR_BITS]; //MODIFICATO page_address_sect
				data_byte_block = 1'b1;
				start_address_block = memory_address[`MEM_ADDR_BITS-1:0]; //MODIFICATO start_address_sect
				byte_ok = 1'b0;
			end

//============================================================================//

/////////--------------- PAGE ERASE INSTRUCTION
			if(page_erase_en == 1'b1)    //receives address in Write Data Bytes Instruction
			begin
				bit_counter_en = 1'b1;
				bit_counter_ld = 1'b1;
				page_erase_en = 1'b0;
				operation = PGER;
				page_address_page = memory_address[`MEM_ADDR_BITS-1:`MEM_ADDR_BITS-`PAGE_ADDR_BITS];
				data_byte_page = 1'b1;
				start_address_page = memory_address[`MEM_ADDR_BITS-1:0];
				byte_ok = 1'b0;
				
			end

//============================================================================//

/////////--------------- SECTOR ERASE INSTRUCTION
			if(sect_erase_en == 1'b1)    //receives address in Write Data Bytes Instruction
			begin
				bit_counter_en = 1'b1;
				bit_counter_ld = 1'b1;
				sect_erase_en = 1'b0;
				operation = ER_SECT;
				page_address_sect = memory_address[`MEM_ADDR_BITS-1:`MEM_ADDR_BITS-`PAGE_ADDR_BITS];
				data_byte_sect = 1'b1;
				start_address_sect = memory_address[`MEM_ADDR_BITS-1:0];
				byte_ok = 1'b0;
			end
		end


//=======================

		//----------------------------data bytes input
		if((mode == active_power_mode)&&(data_byte == 1'b1)&&(byte_ok == 1'b1))
		begin

			if(operation != PGPR_BUFF)//!volatile_reg[1] || (volatile_reg[1]&& (!page_prog_buff_mode_start)))
			begin
				
				byte_ok = 1'b0;
				data_latch[page_offset_addr] = shift_in_reg;
				if(page_offset_addr == {`PAGE_OFFSET_BITS {1'b1}})
				begin
					page_offset_addr = {`PAGE_OFFSET_BITS {1'b0}};
				end
				else
				begin
					//$display("%t MEMORY DEBUG page_offset_addr e' = %d",$realtime, page_offset_addr);
					page_offset_addr = page_offset_addr + {{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
				//$display("%t MEMORY DEBUG adesso page_offset_addr e' = %d",$realtime, page_offset_addr);
				end

				end_address = page_offset_addr;
				bit_counter_en = 1'b1;
				bit_counter_ld = 1'b1;
				write_data_byte_ok = 1'b1;
			end


			else if(operation == PGPR_BUFF)//(volatile_reg[1] && page_prog_buff_mode_start)
			begin
				if(!volatile_reg[0]) //se BUFLD = 0
				begin

					byte_ok = 1'b0;
					//========
					mem_buffer_full = 1'b1;
					//========
					mem_buffer[page_offset_addr_buff] = shift_in_reg;
					if(page_offset_addr_buff == {`PAGE_OFFSET_BITS {1'b1}})
					begin
						page_offset_addr_buff = {`PAGE_OFFSET_BITS {1'b0}};
					end
					else
					begin
						//$display("%t MEMORY DEBUG page_offset_addr_buff e' = %d",$realtime, page_offset_addr_buff);
						page_offset_addr_buff = page_offset_addr_buff + {{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
					//$display("%t MEMORY DEBUG adesso page_offset_addr_buff e' = %d",$realtime, page_offset_addr_buff);
					end

					end_address_buff = page_offset_addr_buff;
					bit_counter_en = 1'b1;
					bit_counter_ld = 1'b1;
					write_data_byte_ok = 1'b1;
				end//if(!volatile_reg[0]
				else if(volatile_reg[0]) //se BUFLD=1
				begin
					$display("%t MEMORY_BUFF:  ERROR: BUFLD = %b cannot write in the buffer. Wait untile Buffer will be free",$realtime,volatile_reg[0]);
				end//else if(volatile_reg[0])
			end//else if(volatile_reg[1] && page_prog_buff_mode_start)
		end//if((mode == active_power_mode)&&(data_byte == 1'b1)&&(byte_ok == 1'b1))

		//----------------------------Set LOCKID input
		if((mode == active_power_mode)&&(set_new_lockid_ok == 1'b1))
		begin
			nonxt_byte = 1'b1;
			set_new_lockid_ok = 1'b0;
			$display("%t: ERROR: Chip select \S must be driven high after the rising edge of Serial Clock after the first data Byte: Lock ID has not been Executed!",$realtime);
		end

		if((mode == active_power_mode)&&(lock_byte == 1'b1)&&(byte_ok == 1'b1))
		begin
			byte_ok = 1'b0;
			
			lock_latch = shift_in_reg;
			lock_byte = 1'b0;
			set_new_lockid_ok = 1'b1;
		end

/////============ WRITE VOLATILE REGISTER ============//////
		if(vol_reg_en == 1'b1)    //receives address in Write Data Bytes Instruction
		begin
			//$display(" vol_reg_en %b ", vol_reg_en);
			bit_counter_en = 1'b1;
			bit_counter_ld = 1'b1;
			vol_reg_en = 1'b0;
			
			new_volatile_byte = 1'b1;
			byte_ok = 1'b0;
			operation = WRVR;

		end

/////====================================================/////

		//----------------------------new SR value input
		if((mode == active_power_mode)&&(new_sr_byte == 1'b1)&&(byte_ok == 1'b1))
		begin

			byte_ok = 1'b0;
			sr_latch = shift_in_reg;
			new_sr_byte = 1'b0;
			write_new_sr_ok = 1'b1;
		end
		//----------------------------Read Data From Memory Array
		if(operation == READ_DATA_OUT)
		begin
//			if(tDiff_clk < 20.0) begin
//				$display("%t MEMORY ERROR! Frequency is %g MHz, the READ DATA SINGLE OUTPUT instruction work at max 50MHz!",$realtime,(1000.0/tDiff_clk));
//				#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid
//				dout = 1'bx;
//			end	
//			else begin			
				if(i==0)
				begin
					if(memory_address[`MEM_ADDR_BITS-1:0] == {`MEM_ADDR_BITS {1'b1}}) //`MEM_ADDR_BITS'h3ff)
						memory_address[`MEM_ADDR_BITS-1:0] = {`MEM_ADDR_BITS {1'b0}};
					else memory_address[`MEM_ADDR_BITS-1:0] = memory_address[`MEM_ADDR_BITS-1:0] + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};// `MEM_ADDR_BITS'h001;
					i = 8;
				end
				#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid

				data_out_buf = memory[memory_address[`MEM_ADDR_BITS-1:0]];
				dout = data_out_buf[i-1];
				i = i-1;
//			end
		end


		//----------------------------Fast Single Read Data From Memory Array
		if(operation == FREAD)
		begin

			if(i==0)
			begin
				if(memory_address[`MEM_ADDR_BITS-1:0] == {`MEM_ADDR_BITS {1'b1}}) //`MEM_ADDR_BITS'h3ff)
					memory_address[`MEM_ADDR_BITS-1:0] = {`MEM_ADDR_BITS {1'b0}};
				else memory_address[`MEM_ADDR_BITS-1:0] = memory_address[`MEM_ADDR_BITS-1:0] + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};// `MEM_ADDR_BITS'h001;
				i = 8;
			end

			#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid
			data_out_buf = memory[memory_address[`MEM_ADDR_BITS-1:0]];


			dout = data_out_buf[i-1];
			//$display("%t MEMORY DEBUG 222:  data_out_buf = %h  memory_add = %h  i = %d ",$realtime,data_out_buf,memory_address,i);
			i = i-1;
		end


		//----------------------------Fast Dual Read Data From Memory Array
		if(operation == FDREAD)
		begin

			if(i==0)
			begin
				if(memory_address[`MEM_ADDR_BITS-1:0] == {`MEM_ADDR_BITS {1'b1}}) //`MEM_ADDR_BITS'h3ff)
					memory_address[`MEM_ADDR_BITS-1:0] = {`MEM_ADDR_BITS {1'b0}};
				else memory_address[`MEM_ADDR_BITS-1:0] = memory_address[`MEM_ADDR_BITS-1:0] + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};// `MEM_ADDR_BITS'h001;
				i = 4;
			end

			#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid
			data_out_buf = memory[memory_address[`MEM_ADDR_BITS-1:0]];

			DQ0 = data_out_buf[2*(i-1)];
			dout = data_out_buf[2*(i-1)+1];
			//$display("%t MEMORY DEBUG 222:  data_out_buf = %h  memory_add = %h  i = %d ",$realtime,data_out_buf,memory_address,i);
			i = i-1;
		end

		//----------------------------Fast Quad Read Data From Memory Array
		if(operation == FQREAD)
		begin

			if(i==0)
			begin
				if(memory_address[`MEM_ADDR_BITS-1:0] == {`MEM_ADDR_BITS {1'b1}}) //`MEM_ADDR_BITS'h3ff)
				begin
					memory_address[`MEM_ADDR_BITS-1:0] = {`MEM_ADDR_BITS {1'b0}};
				end
				else
				begin
					memory_address[`MEM_ADDR_BITS-1:0] = memory_address[`MEM_ADDR_BITS-1:0] + {{`MEM_ADDR_BITS-1 {1'b0}},1'b1};// `MEM_ADDR_BITS'h001;
				end
				i = 2;
			end
			#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid
			data_out_buf = memory[memory_address[`MEM_ADDR_BITS-1:0]];

			DQ0 = data_out_buf[4*(i-1)];
			dout = data_out_buf[4*(i-1)+1];
			DQ2 = data_out_buf[4*(i-1)+2];
			DQ3 = data_out_buf[4*(i-1)+3];
			//$display("%t MEMORY DEBUG 222:  data_out_buf = %h  memory_add = %h  i = %d ",$realtime,data_out_buf,memory_address,i);
			i = i-1;
		end


		//----------------------------Read Data From ID Memory Array or Read lock status
		if(operation == READ_ID_DATA_OUT)
		begin
//			if(tDiff_clk < 20.0) begin
//				$display("%t MEMORY ERROR! Frequency is %g MHz, the READ IDENTIFICATION instruction work at max 50MHz!",$realtime,(1000.0/tDiff_clk));
//				#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid
//				dout = 1'bx;
//			end	
//			else begin
				if(i==0)
				begin
					//if((readidwrap)&&(memory_address[`PAGE_OFFSET_BITS:0] == {`PAGE_OFFSET_BITS {1'b0}}))
					//  $display("%t DEBUG: ERROR:  The number of bytes read has exceeded ID page boundary [%d], unexpected data is read\n",$realtime,page_size);

					if(memory_address[`PAGE_OFFSET_BITS:0] =={{`PAGE_OFFSET_BITS {1'b1}},1'b1})
					begin
						memory_address[`PAGE_OFFSET_BITS:0] = {{`PAGE_OFFSET_BITS {1'b0}},1'b0};
						//   $display("%t MEMORY DEBUG 2:  memory address is at boundary %h ",$realtime, memory_address);
						$display("%t: NOTE:  The number of bytes read has reached ID page boundary [%d]\n",$realtime,2*page_size-1);
						readidwrap = 1'b1;
					end
					else memory_address[`PAGE_OFFSET_BITS:0] = memory_address[`PAGE_OFFSET_BITS:0] + {{`PAGE_OFFSET_BITS {1'b0}},1'b1};
					begin
						//          $display("%t MEMORY DEBUG 3:  memory address is %h ",$realtime, memory_address);
						i = 8;
					end
				end
				#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid
				//$display("%t MEMORY DEBUG 4:  data_out_buf = %h page_address = %h memory_address[] = %h  unione = %h  MEMORY_ID = %h",$realtime,data_out_buf,page_address,memory_address[`PAGE_OFFSET_BITS:0],{page_address,memory_address[`PAGE_OFFSET_BITS:0]}, memory_id[memory_address[`PAGE_OFFSET_BITS:0]]);
				data_out_buf = memory_id[memory_address[`PAGE_OFFSET_BITS:0]];//{page_address,memory_address[`PAGE_OFFSET_BITS:0]}];
				//$display("%t MEMORY DEBUG: memory_address = %h",$realtime, memory_address);
				dout = data_out_buf[i-1];
	//$display("%t MEMORY DEBUG: data_out_buf = %b",$realtime, data_out_buf[i-1]);
	//$display("%t MEMORY DEBUG: dout = %b",$realtime, dout);
				i = i-1;
//			end
		end


//---------------------------------------- Fast Read Data from ID memory Array ----------------//
		if(operation == FRDID)
		begin
			if(i==0)
			begin
				// if((readidwrap)&&(memory_address[`PAGE_OFFSET_BITS-1:0] == {`PAGE_OFFSET_BITS {1'b0}})) $display("%t: ERROR:  The number of bytes read has exceeded ID page boundary [%d], unexpected data is read\n",$realtime,page_size);

				if(memory_address[`PAGE_OFFSET_BITS:0] == {{`PAGE_OFFSET_BITS {1'b1}},1'b1})
				begin
					memory_address[`PAGE_OFFSET_BITS:0] = {{`PAGE_OFFSET_BITS {1'b0}},1'b0};
					$display("%t: NOTE:  The number of bytes read has reached ID page boundary [%d]\n",$realtime,2*page_size-1);
				//readidwrap = 1'b1;
				end
				else memory_address[`PAGE_OFFSET_BITS:0] = memory_address[`PAGE_OFFSET_BITS:0] + {{`PAGE_OFFSET_BITS {1'b0}},1'b1};
				i = 8;
			end
			#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid
			data_out_buf = memory_id[memory_address[`PAGE_OFFSET_BITS:0]];//{page_address,memory_address[`PAGE_OFFSET_BITS:0]}];
			// $display("%t MEMORY DEBUG: memory_address = %h",$realtime, memory_address);
			dout = data_out_buf[i-1];
			i = i-1;
		end

//---------------------------------------- Read_SFDP Register ----------------//
		if(operation == RDSFDP)
		begin
			if(i==0)
			begin

				if(memory_address[`PAGE_OFFSET_BITS-1:0] == {`PAGE_OFFSET_BITS {1'b1}})
				begin
					memory_address[`PAGE_OFFSET_BITS-1:0] = {`PAGE_OFFSET_BITS {1'b0}};
					$display("%t: NOTE:  The number of bytes read has reached SFDP_Register boundary [%d]\n",$realtime,page_size-1);

				end
				else memory_address[`PAGE_OFFSET_BITS-1:0] = memory_address[`PAGE_OFFSET_BITS-1:0] + {{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
				i = 8;
			end
			#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //clock low to Output Valid
			data_out_buf = sfdp_register[memory_address[`PAGE_OFFSET_BITS-1:0]];//[{page_address,memory_address[`PAGE_OFFSET_BITS-1:0]}];
			//$display("%t MEMORY DEBUG: memory_address = %h",$realtime, memory_address);
			dout = data_out_buf[i-1];
			i = i-1;
		end


//---------------------------
		case(instruction)             //execute instruction
//---------------------------
			WRITE_ENABLE:
			begin
				instruction = 5'b0000;
				if(status_reg[0] == 1'b1)   //WIP is set, an internal write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this WRITE ENABLE INSTRUCTION is rejected!",$realtime);
				else begin
					write_en_id = 1'b1;
					operation = EN_WRITE;
				end
			end
//---------------------------
			WRITE_DISABLE:
			begin
				instruction = 5'b0000;
				if(status_reg[0] == 1'b1)   //WIP is set, an internal write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this WRITE DISABLE INSTRUCTION is rejected!",$realtime);
				else begin
					write_dis_id = 1'b1;
					operation = DIS_WRITE;
				end
			end
//---------------------------
			READ_STATUS_REGISTER:
			begin
				t_modify = $realtime;
				modify_operation = 1'b0;
				modify_operation_cher =1'b0;
				operation = READ_SR_OUT;
				if(i==0)  i = 8;
				#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //Clock low to Output Valid
				if (instruction_byte == 1'b0)
					dout = status_reg[i-1];
				else
					dout = 1'bz;

				i = i-1;

			end
//---------------------------
			WRITE_STATUS_REGISTER:
			begin
				instruction = 5'b0000;
				if(status_reg[0] == 1'b1)   //WIP is set, an internal write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this WRITE SR INSTRUCTION is rejected!",$realtime);
				else begin
					if(status_reg[1] == 1'b1) //before "WRITE_SR_IN",WEL must be set
					begin
						sr_write       = 1'b1;
						new_sr_byte    = 1'b1;  //receives new value
						//bit_counter_en = 1'b1;
						//bit_counter_ld = 1'b1;
						bit_counter_en_wr = 1'b1;
						bit_counter_ld_wr = 1'b1;

						operation = WRITE_SR_IN;
					end
					else $display("%t: ERROR: Write Disable! Write SR instruction can not be accepted!",$realtime);
				end
			end
//---------------------------
/////=========== WRITE VOLATILE REGISTER ========/////
			WRITE_VOLATILE_REG:
			begin
				instruction = 5'b00000;
				//$display("%t Ho decodificato la WRVR, WEL = %b ",$realtime,status_reg[1]);
				if(status_reg[0] == 1'b1)   //WIP is set, an internal write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this WRITE VOLATILE INSTRUCTION is rejected!",$realtime);
				else begin
					if(status_reg[1] == 1'b1) //before "VOL_REG_WRITE",WEL must be set
					begin
						//$display("%t SONO QUI e setto new_volatile_byte",$realtime);
						bit_counter_en = 1'b1;
						bit_counter_ld = 1'b1;
						new_volatile_byte = 1'b1;
						//$display("%t MEMORY: vol_reg_en = %b",$realtime, vol_reg_en);
						operation = WRVR;
					//$display("%t sono nella decodifica dell'istruzione: vol_reg_en = %b | new volatile byte = %b  byte_ok = %b ",$realtime,vol_reg_en, new_volatile_byte,byte_ok);
					end
					else $display("%t: ERROR: Write Disable! Write VR instruction can not be accepted!",$realtime);
				end
			end
/////==========================================/////
//---------------------------
/////=========== READ VOLATILE REGISTER ========/////
			READ_VOLATILE_REG:
			begin
				t_modify = $realtime;
				modify_operation = 1'b0;
				modify_operation_cher =1'b0;
				operation = RDVR;
				if(i==0)  i = 8;
				#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);                  //Clock low to Output Valid
				if (instruction_byte == 1'b0)
					dout = volatile_reg[i-1];
				else
					dout = 1'bz;

				i = i-1;

			end
/////==========================================/////
//---------------------------
			READ_FROM_MEMORY_ARRAY:
			begin
				instruction = 5'b0000;      //clear "READ_DATA_BYTES" instruction
				if(status_reg[0] == 1'b1)   //WIP is set, an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this READ FROM MEMORY ARRAY INSTRUCTION is rejected!",$realtime);
				else begin
					read_data_bytes = 1'b1;

					if (`MEM_ADDR_BITS <= 9)
					begin
						address_l_byte = 1'b1; //receives address low byte
						bit_counter_en = 1'b1;
						bit_counter_ld = 1'b1;
					end
					else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
					begin
						address_h_byte  = 1'b1;   //receives address high byte 1
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
					else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
					begin
						address_h2_byte = 1'b1;   //receives address high byte 2
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
				end
			end

			//---------------------------
			FAST_READ_DATA :
			begin
				instruction = 5'b0000;
				if(status_reg[0] == 1'b1)   //WIP is set, an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this READ FROM MEMORY ARRAY INSTRUCTION is rejected!",$realtime);
				else
				begin
					fast_read_data_bytes = 1'b1;

					if (`MEM_ADDR_BITS <= 9)
					begin
						address_l_byte = 1'b1; //receives address low byte
						bit_counter_en = 1'b1;
						bit_counter_ld = 1'b1;
					end
					else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
					begin
						address_h_byte  = 1'b1;   //receives address high byte 1
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
					else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
					begin
						address_h2_byte = 1'b1;   //receives address high byte 2
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
				end
			end


//---------------------------
			FAST_DUAL_READ_DATA :
			begin
				instruction = 5'b0000;
				if(status_reg[0] == 1'b1)   //WIP is set, an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this READ FROM MEMORY ARRAY INSTRUCTION is rejected!",$realtime);
				else
				begin
					fd_read_data_bytes = 1'b1;

					if (`MEM_ADDR_BITS <= 9)
					begin
						address_l_byte = 1'b1; //receives address low byte
						bit_counter_en = 1'b1;
						bit_counter_ld = 1'b1;
					end
					else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
					begin
						address_h_byte  = 1'b1;   //receives address high byte 1
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
					else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
					begin
						address_h2_byte = 1'b1;   //receives address high byte 2
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
				end
			end



//---------------------------
			FAST_QUAD_READ_DATA :
			begin
				instruction = 5'b0000;
				if(status_reg[0] == 1'b1)   //WIP is set, an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this READ FROM MEMORY ARRAY INSTRUCTION is rejected!",$realtime);
				else
				begin
					fq_read_data_bytes = 1'b1;

					if (`MEM_ADDR_BITS <= 9)
					begin
						address_l_byte = 1'b1; //receives address low byte
						bit_counter_en = 1'b1;
						bit_counter_ld = 1'b1;
					end
					else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
					begin
						address_h_byte  = 1'b1;   //receives address high byte 1
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
					else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
					begin
						address_h2_byte = 1'b1;   //receives address high byte 2
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
				end
			end


//---------------------------
			WRITE_TO_MEMORY_ARRAY:
			begin
				instruction = 5'b0000;
				if(status_reg[0] == 1'b1)   //WIP is set, an erase,program,write cycle is in progress
				begin
					$display("%t: WARNING:  An internal write cycle is in progress, this WRITE TO MEMORY ARRAY INSTRUCTION is rejected!",$realtime);

				end
				else begin
					if(status_reg[1] == 1'b1) //before "WRITE DATA",WEL must be set
					begin
						write_data_in  = 1'b1;

						if (`MEM_ADDR_BITS <= 9)
						begin
							address_l_byte = 1'b1; //receives address low byte
							bit_counter_en = 1'b1;
							bit_counter_ld = 1'b1;
						end
						else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
						begin
							address_h_byte = 1'b1;  //receives address high byte 1
							bit_counter_en = 1'b1;
							bit_counter_ld = 1'b1;
						end
						else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
						begin
							address_h2_byte = 1'b1;   //receives address high byte 2
							bit_counter_en  = 1'b1;
							bit_counter_ld  = 1'b1;
						end
					end
					else
					begin
						$display("%t: MEMORY : ERROR: Write Disable! Write instruction can not be accepted!",$realtime);

					end
				end
			end
//---------------------------
			READ_ID_PAGE:
			begin
				instruction = 5'b0000;
				read_id_data_bytes = 1'b1;
				if(status_reg[0] == 1'b1)   //WIP is set, an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this READ FROM IDENTIFICATION PAGE INSTRUCTION is rejected!",$realtime);
				else begin
					if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
					begin
						address_h_byte  = 1'b1;   //receives address high byte 1
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
					else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
					begin
						address_h2_byte = 1'b1;   //receives address high byte 2
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
				end
			end
//---------------------------
//===================== FAST READ ID PAGE ==========================//
			FAST_READ_ID:
			begin
				instruction = 5'b0000;
				fast_read_id_data_bytes = 1'b1;

				if(status_reg[0] == 1'b1)   //WIP is set, an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this FAST READ IDENTIFICATION PAGE INSTRUCTION is rejected!",$realtime);
				else begin
					if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
					begin
						address_h_byte  = 1'b1;   //receives address high byte 1
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
					else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
					begin
						address_h2_byte = 1'b1;   //receives address high byte 2
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
				end
			end
//========================================================//
//---------------------------
			WRITE_ID_PAGE:
			begin
				instruction = 5'b0000;
				if(status_reg[0] == 1'b1)   //WIP is set, an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this WRITE TO MEMORY ARRAY INSTRUCTION is rejected!",$realtime);
				else begin
					if(status_reg[1] == 1'b1) //before "WRITE DATA",WEL must be set
					begin
						write_id_data_in  = 1'b1;

						if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
						begin
							address_h_byte = 1'b1;  //receives address high byte 1
							bit_counter_en = 1'b1;
							bit_counter_ld = 1'b1;
						end
						else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
						begin
							address_h2_byte = 1'b1;   //receives address high byte 2
							bit_counter_en  = 1'b1;
							bit_counter_ld  = 1'b1;
						end
					end
					else $display("%t: ERROR: Write ID Disable! Write instruction can not be accepted!",$realtime);
				end
			end

//---------------------------JEDEC instruction code
			JEDID:
			begin
				t_modify = $realtime;
				modify_operation = 1'b0;
				modify_operation_cher =1'b0;
				operation = JEDEC_ID;
				if(i==0)  i = 24;
				out_jedec = {memory_id[0], memory_id[1],memory_id[2]}; //first three IDPAGE 1 addresses, then the device identification data.
				#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);

				if(instruction_byte == 1'b0)
					dout = out_jedec[i-1];
				else
					dout = 1'bz;
				i= i-1;

			end

//---------------------------READ_CONFIG_SAFETY_REG instruction code
			READ_CONFIG_SAFETY_REG:
			begin
				t_modify = $realtime;
				modify_operation = 1'b0;
				modify_operation_cher =1'b0;
				operation = RDCR;
				if(i==0)  i = 16;
				out_conf_safe_reg = {config_reg,safe_reg};

				#(M95XXX_SIM.M95XXX_Macro_mux.tCLQV);



				if(instruction_byte == 1'b0)
					dout = out_conf_safe_reg[i-1];
				else
					dout = 1'bz;

				i= i-1;

			end

////////===============     Clear Safety Register
			CLEAR_SAFETY_STI_FLAGS:
			begin
				$display("%t: MEMORY: CLEAR SAFETY REGISTER INSTRUCTION is starting!",$realtime);
				instruction = 5'b00000; //4'b0000;
				clear_safe_reg_en = 1'b1;
				instruction_byte = 1'b0;
				byte_ok = 1'b0;
				operation = SAFE_CLEAR_FLAG;

			end
//---------------------------PAGE PROGRAM code
			PAGE_PROGRAM:
			begin

				instruction = 5'b0000;

				if(volatile_reg[1]) //Buffer mode is enabled
				begin : buffer_mode

					//$display("\n%t MEMORY DEBUG: Buffer Mode is active. Page Program in buffer mode is starting",$realtime);
					//$display("%t DEBUG page_prog_buff_mode_start = %b  | !WIP = %b && !BUFLD = %b \n", $realtime, page_prog_buff_mode_start, !status_reg[0],!volatile_reg[0]);

					if(!page_prog_buff_mode_start || ((!status_reg[0] && !volatile_reg[0])) ) //primo accesso alla page program solo se WEL e' attivo
					begin

						$display("\n%t MEMORY: Buffer Mode is active. Page Program in buffer mode is starting",$realtime);

						if(!page_prog_buff_mode_start)
							$display("%t MEMORY: It's first time, the Write Enable is necessary", $realtime);
						//else if ((!status_reg[0] && !volatile_reg[0]))// caso in cui la memoria e' libera e buffer vuoto. quindi scrivo direttamente in memoria.
						//$display("%t MEMORY: Memory is free and buffer is empty! Writing on memory",$realtime);
						//else
						//$display("%t MEMORY: Error",$realtime);
						//$display("%t MEMORY: It's first time, the Write Enable is necessary", $realtime);

						if(status_reg[1] == 1'b1) //WREN
						begin
							if(!page_prog_buff_mode_start)
							begin
								$display("%t MEMORY: Write enabled was applied.",$realtime);
								$display("%t MEMORY: Write enable (WEL) = %b ",$realtime, status_reg[1]);
							end
							if(!safe_reg[5]) //condition on ERF = 0
							begin
								//page_prog_buff_mode_start = 1'b1; //flag che mi indica che ho avuto il primo avvio.
								// la prossima volta che richiamo la page program in buff mode
								// non avro' bisogno del Write Enable.

								//page_program_buff_data_in = 1'b1; //mi fa partire l'operation della page program.
								page_program_data_in = 1'b1;
								if (`MEM_ADDR_BITS <= 9)
								begin
									address_l_byte = 1'b1; //receives address low byte
									bit_counter_en = 1'b1;
									bit_counter_ld = 1'b1;
								end
								else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
								begin
									address_h_byte = 1'b1;  //receives address high byte 1
									bit_counter_en = 1'b1;
									bit_counter_ld = 1'b1;
								end
								else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
								begin
									address_h2_byte = 1'b1;   //receives address high byte 2
									bit_counter_en  = 1'b1;
									bit_counter_ld  = 1'b1;
								end

							end //del safe_reg[5]
						end //del WEL Attivo
					end
					else if(page_prog_buff_mode_start) //secondo accesso in poi.. non bisogna del Write Enable prima
					begin
						//$display("\n%t DEBUG Write enable WEL = %b ",$realtime, status_reg[1]);

						//$display("%t MEMORY DEBUG: Buffer Mode is still active. Another Page Program is starting",$realtime);

						//$display("%t MEMORY_buff DEBUG: WIP = %b   BUFLD = %b\n", $realtime,status_reg[0],volatile_reg[0]);

						if (status_reg[0] && volatile_reg[0]) //WIP = 1, BUFLD = 1  memoria occupata e buffer Pieno.
						begin
							$display("%t: ERROR: Buffer is full and a Page Program is ongoing.\nNew Page Program instruction is not allowed, wait until buffer is free", $realtime);
							safe_reg[4] = 1'b1; //PRF = 1 because has been an error on a program operation
							$display("%t: PRF = %b ",$realtime, safe_reg[4]);
						end

						else  //casi di WIP = 0, BUFLD = 0 memoria libera e buffer libero
						//     e   WIP = 1, BUFLD = 0 memoria occupata e buffer libero
						begin
							if(!safe_reg[5]) //condition on ERF = 0
							begin

								if(!status_reg[0]) //WIP 0, buffer mode active but no program page ongoing
									$display("%t: MEMORY: BUFFER Mode is activated. Page Program instruction is allowed",$realtime);
								else if (status_reg[0] && !volatile_reg[0])
									$display("%t: MEMORY: BUFFER Mode is active and a Page Program instruction is ongoing.\n\t\t    Storing in the Buffer",$realtime);
								//page_program_buff_data_in = 1'b1; //mi fa partire l'operation della page program.
								page_program_data_in = 1'b1;
								//$display("%t MEMORY DEBUG 000 page_program_buff_data_in")
								if (`MEM_ADDR_BITS <= 9)
								begin
									address_l_byte = 1'b1; //receives address low byte
									bit_counter_en = 1'b1;
									bit_counter_ld = 1'b1;
								end
								else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
								begin
									address_h_byte = 1'b1;  //receives address high byte 1
									bit_counter_en = 1'b1;
									bit_counter_ld = 1'b1;
								end
								else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
								begin
									address_h2_byte = 1'b1;   //receives address high byte 2
									bit_counter_en  = 1'b1;
									bit_counter_ld  = 1'b1;
								end

							end //safe_reg[5]
						end//WEL Attivo
					end //else_if page_prog_buff_mode_start
				end //volatile_reg[1]


				else if (!volatile_reg[1])
				begin : normal_mode
					if(status_reg[0] == 1'b1)   //WIP is set, an erase,program,write cycle is in progress
						$display("%t: WARNING:  An internal write cycle is in progress, this WRITE TO MEMORY ARRAY INSTRUCTION is rejected!",$realtime);
					else
					begin
						if(status_reg[1] == 1'b1) //before "PAGE WRITE DATA",WEL must be set
						begin
							if(!safe_reg[5]) //condition on ERF = 0
							begin
								page_program_data_in  = 1'b1;

								if (`MEM_ADDR_BITS <= 9)
								begin
									address_l_byte = 1'b1; //receives address low byte
									bit_counter_en = 1'b1;
									bit_counter_ld = 1'b1;
								end
								else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
								begin
									address_h_byte = 1'b1;  //receives address high byte 1
									bit_counter_en = 1'b1;
									bit_counter_ld = 1'b1;
								end
								else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
								begin
									address_h2_byte = 1'b1;   //receives address high byte 2
									bit_counter_en  = 1'b1;
									bit_counter_ld  = 1'b1;
								end
							end
							else
								$display("%t MEMORY: WARNING previous erase operation was not completely successfull!",$realtime);
						end
						else $display("%t: MEMORY : ERROR: Write Disable! Write instruction can not be accepted!",$realtime);
					end
				end
				else
					$display("%t MEMORY: ERROR Volatile_reg is in an undefined state",$realtime);
			end


//------------------------------- SECTOR ERASE INSTRUCTION ---------//
			SECTOR_ERASE:
			begin
				instruction = 5'b00000;
				if(status_reg[0] == 1'b1)   //if WIP is set (WIP=1), an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this SECTOR ERASE INSTRUCTION is rejected!",$realtime);
				else
				begin
					if(status_reg[1] == 1'b1) //before "SECTOR ERASE",WEL must be set
					begin
						sect_erase_en  = 1'b1;
						if (`MEM_ADDR_BITS <= 9)
						begin
							address_l_byte = 1'b1; //receives address low byte
							bit_counter_en = 1'b1;
							bit_counter_ld = 1'b1;
						end
						else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
						begin
							address_h_byte = 1'b1;  //receives address high byte 1
							bit_counter_en = 1'b1;
							bit_counter_ld = 1'b1;
						end
						else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
						begin
							address_h2_byte = 1'b1;   //receives address high byte 2
							bit_counter_en  = 1'b1;
							bit_counter_ld  = 1'b1;
						end
					end
					else
					begin
						$display("%t: ERROR: Write Disable! SECTOR ERASE instruction can not be accepted!",$realtime);
					end
				end
			end

			//------------------------------- BLOCK ERASE INSTRUCTION ---------//
			BLOCK_ERASE:
			begin
				instruction = 5'b00000;
				if(status_reg[0] == 1'b1)   //if WIP is set (WIP=1), an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this BLOCK ERASE INSTRUCTION is rejected!",$realtime);
				else
				begin
					if(status_reg[1] == 1'b1) //before "BLOCK ERASE",WEL must be set
					begin
						block_erase_en  = 1'b1;
						if (`MEM_ADDR_BITS <= 9)
						begin
							address_l_byte = 1'b1; //receives address low byte
							bit_counter_en = 1'b1;
							bit_counter_ld = 1'b1;
						end
						else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
						begin
							address_h_byte = 1'b1;  //receives address high byte 1
							bit_counter_en = 1'b1;
							bit_counter_ld = 1'b1;
						end
						else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
						begin
							address_h2_byte = 1'b1;   //receives address high byte 2
							bit_counter_en  = 1'b1;
							bit_counter_ld  = 1'b1;
						end
					end
					else
					begin
						$display("%t: ERROR: Write Disable! BLOCK ERASE instruction can not be accepted!",$realtime);
					end
				end
			end

//------------------------------- PAGE ERASE INSTRUCTION ---------//
			PAGE_ERASE:
			begin
				instruction = 5'b00000;

				if(status_reg[0] == 1'b1)   //if WIP is set (WIP=1), an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this PAGE ERASE INSTRUCTION is rejected!",$realtime);
				else
				begin
					if(status_reg[1] == 1'b1) //before "PAGE ERASE",WEL must be set
					begin
						page_erase_en  = 1'b1;
						if (`MEM_ADDR_BITS <= 9)
						begin
							address_l_byte = 1'b1; //receives address low byte
							bit_counter_en = 1'b1;
							bit_counter_ld = 1'b1;
						end
						else if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
						begin
							address_h_byte = 1'b1;  //receives address high byte 1
							bit_counter_en = 1'b1;
							bit_counter_ld = 1'b1;
						end
						else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
						begin
							address_h2_byte = 1'b1;   //receives address high byte 2
							bit_counter_en  = 1'b1;
							bit_counter_ld  = 1'b1;
						end
					end
					else
					begin
						$display("%t: ERROR: Write Disable! PAGE ERASE instruction can not be accepted!",$realtime);
					end
				end
			end


//------------------------------- CHIP ERASE INSTRUCTION ---------//
			CHIP_ERASE:
			begin
				instruction = 5'b00000;
				if(status_reg[0] == 1'b1)   //WIP is set, an internal write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this CHIP ERASE INSTRUCTION is rejected!",$realtime);
				else begin
					if(status_reg[1] == 1'b1) //before "CHIP ERASE",WEL must be set
					begin
						bit_counter_en = 1'b1;
						bit_counter_ld = 1'b1;
						chip_erase_en = 1'b1;
						operation = CHER;
					end
					else $display("%t: ERROR: Write Disable! Chip Erase instruction can not be accepted!",$realtime);
				end
			end

//----------- READ SFDP_Register -------------------//
			READ_SFDP:
			begin
				instruction = 5'b0000;
				sfdp_data_bytes = 1'b1;
				if(status_reg[0] == 1'b1)   //WIP is set, an erase,program,write cycle is in progress
					$display("%t: WARNING:  An internal write cycle is in progress, this READ SFDP_Register INSTRUCTION is rejected!",$realtime);
				else begin
					if (9 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 16)
					begin
						address_h_byte  = 1'b1;   //receives address high byte 1
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
					else if (16 < `MEM_ADDR_BITS && `MEM_ADDR_BITS <= 24)
					begin
						address_h2_byte = 1'b1;   //receives address high byte 2
						bit_counter_en  = 1'b1;
						bit_counter_ld  = 1'b1;
					end
				end
			end

//------------------RESET ENABLE--------//////
			RESET_ENABLE:
			begin
				instruction = 5'b0000;

				reset_en_id = 1'b1;

				operation = RSTEN;


			end

/////------- SOFTWARE RESET ------/////
			RESET_SOFTWARE:
			begin
				instruction = 5'b00000;
				if(reset_en_bit) //before "SOFTWARE RESET",RESET ENABLE must be set
				begin
					bit_counter_en = 1'b1;
					bit_counter_ld = 1'b1;
					res_soft_en = 1'b1;
					operation = RESET;
				end
				else $display("%t: ERROR: RESET ENABLE Disable! SOFTWARE RESET instruction can not be accepted!",$realtime);

			end

//------------------ DEEP POWER-DOWN--------//////
			DEEP_POWER_ENTER:
			begin
				instruction = 5'b0000;
				dpd_en = 1'b1;
				operation = DPD;

			end
/////==========================================/////

//------------------ RELEASE DEEP POWER-DOWN--------//////
			RELEASE_DEEP_POWER:
			begin
				instruction = 5'b0000;
				rdpd_en = 1'b1;
				operation = RDPD;

			end
/////==========================================/////

		endcase
	end

	always@(posedge S)
	begin : secondo_posedgeS
		

		//----------------------------Clear Safety Register (CLRSF) Instruction Execute
		if(operation == SAFE_CLEAR_FLAG)
		begin

			modify_operation =1'b1;
			modify_operation_cher =1'b0;
			operation = 5'b00000;
			if(clear_safe_reg_en == 1'b1)
			begin
				clear_safe_reg_en = 1'b0;
				safe_reg[7:0] = 8'b00000000;
				$display("%t: NOTE: CLEAR SAFETY REGISTER.\n",$realtime);
				$display("%t: MEMORY: the safety register is reset [%h]", $realtime, safe_reg);
			end
			else if(clear_safe_reg_en == 1'b0)
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of the write enable instruction code is latched. Clear Safety Register instruction is not executed.\n",$realtime);
		end



		/////====================================== WRITE VOLATILE REGISTER =======/////
		if(operation == WRVR)
		begin
			operation = 5'b00000;
			modify_operation =1'b1;
			modify_operation_cher =1'b0;
			//$display("%t MEMORY DEBUG 1: vol_reg_en = %b",$realtime, vol_reg_en);
			if(vol_reg_en == 1'b1)
			begin
				vol_reg_en = 1'b0;
				$display("%t: NOTE: Begin to Write Volatile Register!",$realtime);
				status_reg[0] = 1'b1;           //WIP is 1 during this cycle
				//#(M95XXX_SIM.M95XXX_Macro_mux.tPW);

				//added for page program
				page_prog_buff_mode_start = 1'b0;



				volatile_reg[1] = volatile_latch[1];
				volatile_reg[0] = ~volatile_reg[1];
				$display("%t: NOTE : Write Volatile Register successfully! Now it is %b ",$realtime, volatile_reg);
				$display("%t: NOTE : BUFLD bit is set to %d because it is the negate of BUFEN\n",$realtime, volatile_reg[0]);
				$display("%t: NOTE : BUFEN bit is set to %d", $realtime, volatile_reg[1]);
				status_reg[0] = 1'b0;           //WIP is reset when write cycle is completed.
				status_reg[1] = 1'b0;           //WEL is reset when write cycle is completed.
				for(i=2;i<=7;i=i+1)
				begin
					$display("%t: ERROR : Bit[%d] can not be written but it is accessed in only-read mode\n",$realtime,i);
				end
				i=2;
			end
			else
				$display("%t: ERROR: \"/S\" should not be deselected after the 8th bit of data is latched, Write Volatile instruction is not executed.\n",$realtime);
		end

		if(   (page_program_data_end)  && (volatile_reg[1]) && (!page_prog_buff_mode_start)) //(operation == PGPR)  && (write_data_byte_ok) &&
		begin

			#50;
			//  $display("\t DEBUG==========================================");
			//  $display("%t MEMORY DEBUG: ho finito la prima run della Program Page in buff mode. Setto il flag page_prog_buff_mode_start = %b",$realtime,page_prog_buff_mode_start);
			//  $display("\t DEBUG==========================================");
			page_prog_buff_mode_start = 1'b1;
		//  $display("%t MEMORY DEBUG: page_prog_buff_mode_start = %b",$realtime,page_prog_buff_mode_start);
		end

	end


	always@(mem_buffer_full , posedge S )  //PROCESSO PER ALZARE IL BUFLD!
	begin
		
		volatile_reg[0] = S && mem_buffer_full;
		//$display("%t DEBUG volatile_reg[0] %b",$realtime, volatile_reg[0]);
	end

	always@(posedge volatile_reg[0])
	begin
		
		if(volatile_reg[1])
		begin : Writing_buffer_data_to_memory

			operation = 4'b00000;

			//AGGIUNTO DOPO DA VERIFICARE
			page_no_buff = page_address_buff;
			if(status_reg[4:2] == 3'b111) //regardeless of TB, the whole memory is write protected
			begin
				$display("%t: WARNING: Whole memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
				safe_reg[7] = 1'b1; //PAMAF = 1
				safe_reg[4] = 1'b1; //PRF
				volatile_reg[0] = 1'b0;
				mem_buffer_full = 1'b0;

			end
			else if ((status_reg[6]) && (status_reg[4:2] != 3'b000)&&(page_no_buff <= `PAGES/2))//TB = 1  the memory is blocked from lower side
			begin
//                              safe_reg[7] = 1'b1; //PAMAF = 1
//                              safe_reg[4] = 1'b1; //PRF

				if ((status_reg[4:2] == 3'b110)&&(page_no_buff <= `PAGES/2)&&(!write_id_data_end)) //half memory is write protected, from 000000h to 1FFFFFh 2MB
				begin
					$display("%t: WARNING: Lower half of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 1FFFFFh are write protected, then address %h is not writable", memory_address_buff);//page_no);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
				else if((status_reg[4:2] == 3'b101)&&(page_no_buff <= `PAGES/4)&&(!write_id_data_end)) //a quarter of memory is write protected  1MB
				begin
					$display("%t: WARNING: Lower quarter of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 0FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
				else if((status_reg[4:2] == 3'b100)&&(page_no_buff <= `PAGES/8)&&(!write_id_data_end)) // 1/8 of memory is write protected  512KB
				begin
					$display("%t: WARNING: Lower eight of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 07FFFFhh are write protected, then address %h is not writable",  memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
				else if((status_reg[4:2] == 3'b011)&&(page_no_buff <= `PAGES/16)&&(!write_id_data_end)) //1/16 of memory is write protected 256KB
				begin
					$display("%t: WARNING: Lower sixteenth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 03FFFFh are write protected, then address %h is not writable",  memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
				else if((status_reg[4:2] == 3'b010)&&(page_no_buff <= `PAGES/32)&&(!write_id_data_end)) //1/32 of memory is write protected 128KB
				begin
					$display("%t: WARNING: Lower thirty-second of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 01FFFFh are write protected, then address %h is not writable",  memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
				else if((status_reg[4:2] == 3'b001)&&(page_no_buff <= `PAGES/64)&&(!write_id_data_end)) //1/64 of memory is write protected  64KB
				begin
					$display("%t: WARNING: Lower sixty-fourth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 00FFFFh are write protected, then address %h is not writable",  memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
			end
			else if ((!status_reg[6]) && (status_reg[4:2] != 3'b000)&&(page_no_buff >= `PAGES/2)) //TB =0  memory is blocked from upper side
			begin
//                          safe_reg[7] = 1'b1; //PAMAF = 1
//                          safe_reg[4] = 1'b1; //PRF

				if ((status_reg[4:2] == 3'b110)&&(page_no_buff >= `PAGES/2)&&(!write_id_data_end)) //half memory is write protected, from 200000h to 3FFFFFh 2MB
				begin
					$display("%t: WARNING: upper half of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 200000h to 3FFFFFh are write protected, then address %h is not writable", memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
				else if((status_reg[4:2] == 3'b101)&&(page_no_buff >= (`PAGES - (`PAGES/4)))&&(!write_id_data_end)) //a quarter of memory is write protected  1MB
				begin
					$display("%t: WARNING: Lower quarter of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
				else if((status_reg[4:2] == 3'b100)&&(page_no_buff >= (`PAGES - (`PAGES/8)))&&(!write_id_data_end)) // 1/8 of memory is write protected  512KB
				begin
					$display("%t: WARNING: Lower eight of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
				else if((status_reg[4:2] == 3'b011)&&(page_no_buff >= (`PAGES - (`PAGES/16)))&&(!write_id_data_end)) //1/16 of memory is write protected    256KB
				begin
					$display("%t: WARNING: Lower sixteenth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
				else if((status_reg[4:2] == 3'b010)&&(page_no_buff >= (`PAGES - (`PAGES/32)))&&(!write_id_data_end)) //1/32 of memory is write protected 128KB
				begin
					$display("%t: WARNING: Lower thirty-second of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
				else if((status_reg[4:2] == 3'b001)&&(page_no_buff >= (`PAGES - (`PAGES/64)))&&(!write_id_data_end)) //1/64 of memory is write protected  64KB
				begin
					$display("%t: WARNING: Lower sixty-fourth of memory is Write Protected, the Byte Write instruction is not accepted!\n",$realtime);
					$display(" Addresses from 000000h to 3FFFFFh are write protected, then address %h is not writable",  memory_address_buff);
					safe_reg[7] = 1'b1; //PAMAF = 1
					safe_reg[4] = 1'b1; //PRF
					volatile_reg[0] = 1'b0;
					mem_buffer_full = 1'b0;
				end
			end
			else //memory is not protected
			begin



				$display("%t MEMORY: Waiting memory is free",$realtime);

				wait(status_reg[0] == 1'b0); //WIP 0

				$display("%t MEMORY: Just finished writing in memory. Memory is free | WIP = %b ",$realtime, status_reg[0]);

				status_reg[0] = 1'b1; //WIP 1
				data_byte = 1'b0;
				//$display("%t MEMORY DEBUG: Loading Buffer Data   | WIP = %b", $realtime, status_reg[0]);

				start_address_chk = {start_address_buff[8:4],4'b0000}; //start check from the beginning of the 16 bits

				if(end_address_buff[3:0] != 4'b0000)
				begin
					//$display(" DEBUG 3");

					end_address_chk = end_address_buff +(5'b10000 - end_address_buff[3:0]); //end the check at the end of the 16 bits
					//$display("%t DEBUG end_address_chk = %b |  end_address_buff = %b ",$realtime, end_address_chk,end_address_buff);
					if (end_address_chk[3:0] == 4'b0000)
						$display("%t MEMORY: Check if addresses from [%h] to [%h] are in Erase Status",$realtime,{page_address_buff,start_address_chk},{page_address_buff,end_address_chk-1'h1});
					else
						$display("%t MEMORY: Check if addresses from [%h] to [%h] are in Erase Status",$realtime,{page_address_buff,start_address_chk},{page_address_buff,end_address_chk});

				end
				else
				begin
					end_address_chk = end_address_buff; //the end address is the last one of the 16 bits.
					$display("%t MEMORY: Check if addresses from [%h] to [%h] are in Erase Status",$realtime,{page_address_buff,start_address_chk},{(page_address_buff),end_address_chk-1'h1});
				end

				if (start_address_chk == end_address_chk)
				begin : loop_erase_chk_3
					//$display("%t DEBUG erased_status_chk %h ",$realtime,erased_status_chk);
					for(idx=1;idx<=`PAGE_SIZE;idx=idx+1)
					begin
						erased_status_chk = erased_status_chk & memory[{page_address_buff,start_address_chk}];
						//$display("%t MEMORY DEBUG: erased_status_chk %h | data in memory %h", $realtime, erased_status_chk,memory[{page_address,start_address_chk}]);
						//$display("%t MEMORY DEBUG: erased_status_chk %h | data in memory %h | addr = %b [%h]", $realtime, erased_status_chk,memory[{page_address_buff,start_address_chk}],{page_address_buff,start_address_chk},{page_address_buff,start_address_chk});

						if(erased_status_chk != 8'hFF)
						begin
							//$display("%t MEMORY DEBUG: Page program operation failed, the selected memory addresses are already written. Do an erase instruction before. ",$realtime);
							$display("%t MEMORY: Page Program Failed because some bytes between %h and %h are not in erase state. Do an erase instruction before.",$realtime,{page_address_buff,start_address_chk},{page_address_buff,end_address_chk});
							safe_reg[4]= 1'b1; //PRF = 1;
							$display("%t: PRF = %b ",$realtime, safe_reg[4]);
							disable Writing_buffer_data_to_memory;
							disable loop_erase_chk_3;
						end
						if(start_address_chk == {`PAGE_OFFSET_BITS {1'b1}})
						begin
							start_address_chk = {`PAGE_OFFSET_BITS {1'b0}};
						end
						else
							start_address_chk = start_address_chk + 1; //{{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};

					end
				end //: loop_erase_chk
				else
				begin : loop_erase_chk4
					//$display(" DEBUG loop erase chk4 file finale");
					//$display("%t DEBUG start_address_chk %h [%b]| end_address_chk %h [%b]",$realtime,start_address_chk,start_address_chk,end_address_chk,end_address_chk);
					//$display("%t DEBUG erased_status_chk %h ",$realtime, erased_status_chk);
					while(!(start_address_chk == end_address_chk))
					begin
						erased_status_chk = erased_status_chk & memory[{page_address_buff,start_address_chk}];
						//$display("%t MEMORY DEBUG: start_address_chk %h [%b] | end_address %h [%b]", $realtime,start_address_chk,start_address_chk,end_address,end_address);
						//$display("%t MEMORY DEBUG: erased_status_chk %h | data in memory %h", $realtime, erased_status_chk,memory[{page_address,start_address_chk}]);
						//$display("%t MEMORY DEBUG: erased_status_chk %h | data in memory %h | addr = %b [%h]", $realtime, erased_status_chk,memory[{page_address_buff,start_address_chk}],{page_address_buff,start_address_chk},{page_address_buff,start_address_chk});

						if(erased_status_chk != 8'hFF)
						begin
							$display("%t MEMORY: Page Program Failed because some bytes between %h and %h are not in erase state. Do an erase instruction before.",$realtime,{page_address,start_address_chk},{page_address,end_address_chk});
							safe_reg[4]= 1'b1; //PRF = 1;
							$display("%t: PRF = %b ",$realtime, safe_reg[4]);
							disable Writing_buffer_data_to_memory;
							disable loop_erase_chk4;
						end
						if(start_address_chk == {`PAGE_OFFSET_BITS {1'b1}})
						begin
							start_address_chk = {`PAGE_OFFSET_BITS {1'b0}};
						end
						else
							start_address_chk = start_address_chk + 1; //{{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
					end //del while
				end //: loop_erase_chk2


				if(end_address_buff[3:0] != 4'b0000)
				begin
					if (end_address_chk[3:0] == 4'b0000)
						$display("\n%t MEMORY: Memory addresses from [%h] to [%h] are in erase status. Page Progam cycle is starting",$realtime,{page_address_buff,start_address_buff[8:4],4'b0000},{(page_address_buff),end_address_chk-1'h1});//{start_address[8:4],4'b0000},end_address_chk);

					else
						$display("\n%t MEMORY: Memory addresses from [%h] to [%h] are in erase status. Page Progam cycle is starting",$realtime,{page_address_buff,start_address_buff[8:4],4'b0000},{(page_address_buff),end_address_chk});//{start_address[8:4],4'b0000},end_address_chk);


				//$display("%t MEMORY DEBUG: Memory addresses from [%h] to [%h] are in erase status. Page Progam cycle is starting",$realtime,{page_address_buff,start_address_buff[8:4],4'b0000},{(page_address_buff),end_address_chk});//{start_address[8:4],4'b0000},end_address_chk);
				end
				else
				begin
					$display("\n%t MEMORY: Memory addresses from [%h] to [%h] are in erase status. Page Progam cycle is starting",$realtime,{page_address_buff,start_address_buff[8:4],4'b0000},{(page_address_buff),end_address_chk-1'h1});//{start_address[8:4],4'b0000},end_address_chk);
				end
				//$display("%t MEMORY DEBUG: Memory addresses from [%h] to [%h] are in erase status. Writing the datas in the data latch",$realtime,{page_address_buff,start_address_buff[8:4],4'b0000},{page_address_buff,end_address_chk});//{start_address[8:4],4'b0000},end_address_chk);


				// ==================================================================================================================================
				$display("%t MEMORY: Loading Data from buffer to data latch!", $realtime);

				//$display("%t  DEBUG SCRIVO i dati del buffer nel data_latch fino all'indirizzo %d ",$realtime, end_address_buff-1);

				memory_address = memory_address_buff;
				end_address = end_address_buff;
				start_address = start_address_buff;
				start_address_copy = start_address_buff;
				page_address = page_address_buff;
				//data_byte = 1'b0;

				num_bytes = (end_address - start_address)== 0 ? 512 : (end_address-start_address) ;
				incr = 0;
				while(incr < num_bytes)
				begin


					//$display("%t  DEBUG sto scrivendo l'indirizzo %h  = %h ",$realtime,start_address_copy+incr,{page_address_buff,start_address_copy});
					//$display("%t  DEBUG sto scrivendo l'indirizzo  %h (%d di %d) ",$realtime,start_address_copy,incr,num_bytes);
					//data_latch[start_address_copy] = mem_buffer[start_address_copy];
					data_latch[start_address_copy] = mem_buffer[start_address_copy];
					start_address_copy = start_address_copy + 1;
					incr = incr +1;

				end

				$display("%t MEMORY: Data wrote in data latch then memory buffer is empty",$realtime);

				volatile_reg[0] = 1'b0;
				mem_buffer_full = 1'b0;
				//$display("%t DEBUG vol_reg[0] = %b",$realtime,volatile_reg[0]);
				//  ===================================================================================================================================

				$display("%t MEMORY: Writing in memory: 1.5ms from this moment! ",$realtime);
				#(M95XXX_SIM.M95XXX_Macro_mux.tPP); //1.5 ms
				
				if (erased_status_chk == 8'hFF)
				begin
					//                      $display("%t MEMORY DEBUG: WRITING .....................................",$realtime);
					//                      $display("%t MEMORY DEBUG: start_address = %h | end_address = %h",$realtime,start_address,end_address);
					if(start_address == end_address)  //Definite Page wrap around
					begin : loop8
							
						for(idx=1;idx<=`PAGE_SIZE;idx=idx+1)
						begin
							//                                      $display("%t MEMORY DEBUG loop8  idx = %d",$realtime, idx);
							//                                      $display("%t MEMORY DEBUG loop8: memory data = %h in the address %h ",$realtime, memory[{page_address,start_address}],{page_address,start_address});
							memory[{page_address,start_address}] = data_latch[start_address];
							//                                      $display("%t MEMORY DEBUG loop8 :now data in memory in the address [%h]  is %h : data_latch [%h]  ", $realtime,{page_address,start_address}, memory[{page_address,start_address}],data_latch[start_address]);

							if(start_address == {`PAGE_OFFSET_BITS {1'b1}})
							begin
								start_address = {`PAGE_OFFSET_BITS {1'b0}};

							end
							else start_address = start_address + {{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
						end //del for
					end //del begin : loop5
					else
					begin :loop9
						while(!(start_address == end_address))
						begin
							
							//$display("%t MEMORY DEBUG loop9: start_address = %h | end_address = %h",$realtime,start_address,end_address);
							//$display("%t MEMORY DEBUG loop9: memory data = %h in the address %h ",$realtime, memory[{page_address,start_address}],{page_address,start_address});
							memory[{page_address,start_address}] = data_latch[start_address];
							//$display("%t MEMORY DEBUG loop9 :now data in memory in the address [%h]  is %h : data_latch [%h]  ", $realtime,{page_address,start_address}, memory[{page_address,start_address}],data_latch[start_address]);

							if(start_address == {`PAGE_OFFSET_BITS {1'b1}})
							begin
								start_address = {`PAGE_OFFSET_BITS {1'b0}};
							end
							else
								start_address = start_address + 1; //{{`PAGE_OFFSET_BITS-1 {1'b0}},1'b1};
						end //del while
					end //dell'begin : loop6   */


					$display("%t WIP = %b ",$realtime,status_reg[0]);
					//status_reg[0] = 1'b0; //WIP 0



					$display("%t ++++++++++++++++++++++++++++++++++++++++++++++++++++++++",$realtime);
					$display("%t MEMORY:  NOTE: Page Program Buffer Mode Operation is finished!! ",$realtime);
					$display("%t ++++++++++++++++++++++++++++++++++++++++++++++++++++++++ \n \n ",$realtime);

					page_program_buff_data_end = 1'b0;
				end
				else
				begin
					$display("%t MEMORY: Page Program Failed because some bytes between %h and %h are not in erase state",$realtime,start_address,end_address);
				end
			end
			status_reg[0] = 1'b0; //WIP 0
		end//Writing_buffer_data_to_memory

	end //always@(volatile_reg[0)






	always@(negedge volatile_reg[1])
	begin
		$display("%t MEMORY: Buffer mode has been deactivated ",$realtime);
		status_reg[1] = 1'b0;
		page_prog_buff_mode_start = 1'b0;
	end



//	always @(posedge page_program_buff_data_end)
//			$display(" %t DEBUG page_program_buff_data_end = %b",$realtime,page_program_buff_data_end);
//	
//	always @(negedge page_program_buff_data_end)
//			$display(" %t DEBUG page_program_buff_data_end = %b",$realtime,page_program_buff_data_end);




//=========================================================
//AC timing Characteristics Check
//=========================================================
	always@(posedge S)    //rising edge of /S
	begin
		r_S = 1'b1; t_rS = $realtime;
		if(r_C == 1'b1)
		begin
			tCHSH = t_rS - t_rC;
			if(tCHSH < M95XXX_SIM.M95XXX_Macro_mux.tCHSH) $display("%t: ERROR: /S Active Hold Time(tCHSH) violated.\n",$realtime);
		end
	end
//================================
	always@(negedge S)    //falling edge of /S
	begin
		f_S = 1'b1; t_fS = $realtime;
		if(r_S == 1'b1)
		begin               //check /S deselect time
			tSHSL = t_fS - t_rS;
			if(tSHSL < M95XXX_SIM.M95XXX_Macro_mux.tSHSL) $display("%t: ERROR: /S Deselect Time(tSHSL) violated.\n",$realtime);
		end
		if(r_C == 1'b1)
		begin               //check /S not active hold time (relative to clk_in)
			tCHSL = t_fS - t_rC;
			if(tCHSL < M95XXX_SIM.M95XXX_Macro_mux.tCHSL) $display("%t: ERROR: /S Not Active Hold Time(tCHSL) violated.\n",$realtime);
		end
	end
//================================
	always@(C)
	begin
		if(C == 1'b1)       //rising edge of clock
		begin
			t_rC = $realtime;
			if(r_C == 1'b1)
			begin
				Tc = t_rC - t_rC1;
				if(Tc < $floor((M95XXX_SIM.M95XXX_Macro_mux.tC)*100)/100.0) $display("%t: ERROR: Clock Frequency(fC) violated, fC > %4.2f MHz. Tc=%3.10g TC=%3.10g\n",$realtime,M95XXX_SIM.M95XXX_Macro_mux.fC,Tc, M95XXX_SIM.M95XXX_Macro_mux.tC);
			end
			r_C = 1'b1;
			if(f_S == 1'b1)
			begin
				tSLCH = t_rC - t_fS;
				if(tSLCH < M95XXX_SIM.M95XXX_Macro_mux.tSLCH) $display("%t: ERROR: /S Active Setup Time(tSLCH) violated.\n",$realtime);
			end
			if(r_S == 1'b1)
			begin
				tSHCH = t_rC - t_rS;
				if(tSHCH < M95XXX_SIM.M95XXX_Macro_mux.tSHCH) $display("%t: ERROR: /S Not Active Setup Time(tSHCH) violated.\n",$realtime);
			end
			if(f_C == 1'b1)
			begin
				tCL = t_rC - t_fC;
				if(tCL < M95XXX_SIM.M95XXX_Macro_mux.tCL) $display("%t: ERROR: Clock Low Time(tCL) violated.\n",$realtime);
			end
			if(din_change == 1'b1)
			begin
				tDVCH = t_rC - t_d;
				if(tDVCH < M95XXX_SIM.M95XXX_Macro_mux.tDVCH) $display("%t: ERROR: Data In Setup Time(tDVCH) violated.\n",$realtime);
			end
			if(f_H == 1'b1)
			begin
				tHLCH = t_rC - t_fH;
				if(tHLCH < M95XXX_SIM.M95XXX_Macro_mux.tHLCH) $display("%t: ERROR: Clock Low Hold Time After HOLD Active(tHLCH) violated.\n",$realtime);
			end
			if(r_H == 1'b1)
			begin
				tHHCH = t_rC - t_rH;
				if(tHHCH < M95XXX_SIM.M95XXX_Macro_mux.tHHCH) $display("%t: ERROR: Clock Low Hold Time After HOLD Not Active(tHHCH) violated.\n",$realtime);
			end
			t_rC1 = t_rC;
		end
		//================================
		if(C == 1'b0)       //falling edge of clock
		begin
			f_C = 1'b1; t_fC = $realtime;
			if(r_C == 1'b1)
			begin
				tCH = t_fC - t_rC;
				if(tCH < M95XXX_SIM.M95XXX_Macro_mux.tCH) $display("%t: ERROR: Clock High Time(tCH) violated.\n",$realtime);
			end
		end
	end
//================================
	always@(posedge HOLD)
	begin
		r_H = 1'b1;
		t_rH = $realtime;
		if(f_C == 1'b1)
		begin
			tCLHH = t_rH - t_fC;
			if(tCLHH < M95XXX_SIM.M95XXX_Macro_mux.tCLHH) $display("%t: ERROR: Clock High Set-Up Time Before HOLD Not Active(tCLHH) violated.\n",$realtime);
		end
	end
//================================
	always@(negedge HOLD)
	begin
		f_H = 1'b1;
		t_fH = $realtime;
		if(f_C == 1'b1)
		begin
			tCLHL = t_fH - t_fC;
			if(tCLHL < M95XXX_SIM.M95XXX_Macro_mux.tCLHL) $display("%t: ERROR: Clock High Set-Up Time Before HOLD Active(tCLHL) violated.\n",$realtime);
		end
	end
//================================
	always@(D)            //Input data change on "D" line
	begin
		din_change = 1'b1;
		t_d = $realtime;
		if(r_C == 1'b1)
		begin
			tCHDX = t_d - t_rC;
			if(tCHDX < M95XXX_SIM.M95XXX_Macro_mux.tCHDX) $display("%t: ERROR: Data In Hold Time(tCHDX) violated.\n",$realtime);
		end
	end
//================================

endmodule
//pragma protect end
